const pg = require('../General/Model');
const multer = require('multer');
const path = require('path');
const fs = require('fs');
// Database Query Functions
class BannerDatabase {
    // Create Banner
    static async createBanner(type, imagePath) {
        const query = `
            INSERT INTO banners (type, image)
            VALUES ($1, $2)
            RETURNING *
        `;
        try {
            const result = await pg.query(query, [type, imagePath]);
            return result.rows[0];
        } catch (error) {
            console.error('Error creating banner:', error);
            throw error;
        }
    }

    // Get Banner by Type
    static async getBannerByType(type) {
        const query = `
    SELECT 
        id,
        type,
        image,
        created_at,
        updated_at
    FROM banners 
    WHERE type = $1 
    ORDER BY created_at DESC 
    LIMIT 1
`;
        try {
            const result = await pg.query(query, [type]);
            return result.rows[0] || null;
        } catch (error) {
            console.error('Error getting banner by type:', error);
            throw error;
        }
    }

    // Get All Banners
    static async getAllBanners() {
     const query = `
    SELECT 
        id,
        type,
        image,
        created_at,
        updated_at
    FROM banners 
    ORDER BY created_at DESC
`;
        try {
            const result = await pg.query(query);
            return result.rows;
        } catch (error) {
            console.error('Error getting all banners:', error);
            throw error;
        }
    }

    // Update Banner
    static async updateBanner(type, newImagePath) {
        try {
            // First, find the existing banner to get the old image path
            const existingBanner = await this.getBannerByType(type);

            if (existingBanner && existingBanner.image) {
                const oldImagePath = path.join(__dirname, 'banners', existingBanner.image);
                
                // Delete the old image file if it exists
                if (fs.existsSync(oldImagePath)) {
                    fs.unlinkSync(oldImagePath);
                    console.log(`Deleted old image: ${oldImagePath}`);
                }
            }

            // Update the banner in the database
            const query = `
                UPDATE banners 
                SET image = $2, updated_at = CURRENT_TIMESTAMP
                WHERE type = $1
                RETURNING *
            `;
            
            const result = await pg.query(query, [type, newImagePath]);
            return result.rows[0] || null;
        } catch (error) {
            console.error('Error updating banner:', error);
            throw error;
        }
    }

    static async getAllImagesBinary() {
        const query = `
            SELECT id, type, image 
            FROM banners 
            ORDER BY created_at DESC
        `;
        try {
            const result = await pg.query(query);
            
            // Map to store binary images
            const imageBinaries = [];

            for (const banner of result.rows) {
                if (banner.image) {
                    const imagePath = path.join(__dirname, 'banners', banner.image);
                    
                    try {
                        // Read file as binary
                        const imageBuffer = fs.readFileSync(imagePath);
                        
                        imageBinaries.push({
                            id: banner.id,
                            type: banner.type,
                            filename: banner.image,
                            binary: imageBuffer
                        });
                    } catch (fileError) {
                        console.error(`Error reading image file ${banner.image}:`, fileError);
                    }
                }
            }

            return imageBinaries;
        } catch (error) {
            console.error('Error getting all image binaries:', error);
            throw error;
        }
    }
}

module.exports = BannerDatabase;