const _ = require('lodash');
const Rule = require('../Rule');
const Result = require('./Result');
const Queue = require('../../General/Queue');
const { encode } = require('../../General/Buffer');
const C = require('../../General/Constant');
const H = require('../../General/Helper');

class Hilo {
	constructor(io, client, data, id) {
		this.io = io
		this.client = client
		this.data = data
		this.id = _.toNumber(id)
	}

	/*
	   * Send Data To Client
	*/
	emit(data) {
		this.client.emit(C.PLAY_HILO, encode(data))
	}

	/*
	 * Intialize Game
	*/
	init() {

		let { command } = this.data;

		switch (command) {
			case 'play':
				return this.play();

			case 'cashout':
				return this.end(false);

			case 'next':
				return this.next();

			case 'high':
				return this.high();

			case 'low':
				return this.low();
		}
	}

	/*
	 * Play Game
	*/
	play() {
		let self = this;

		Rule.CanPlay(self.id, self.data, self.client, 'hilo', (status, err) => {
			if (status !== true) {
				return this.emit({ command: 'error', uid: self.id, message: status, code: err })
			}

			Rule.preparePlay(self.client, self.id, 'hilo', self.data, (isOk) => {
				if (!isOk) {
					console.log('Error Hilo Playing: 21')
					return false;
				}

				//Generate Random Hash and Result
				let randomResult = Result.make();
				let hash = randomResult.hash;
				let result = randomResult.result;

				//Update Queue and Save Result and Hash
				Queue.update(self.id, 'uid', 'result', result);
				Queue.update(self.id, 'uid', 'hash', hash);
				Queue.update(self.id, 'uid', 'profit', 0);
				Queue.update(self.id, 'uid', 'next', 0);
				Queue.update(self.id, 'uid', 'round', 0);

				//Send Play Game
				this.emit({ command: 'play', hash: hash, target: result[0] })
			})
		})
	}

	/*
	 * Choose Next / Skip
	*/
	next() {
		let { selected } = this.data;
		let { result, next } = Queue.getOne('uid', this.id);

		next++

		this.emit({ command: 'next', target: result[next] })
		Queue.update(this.id, 'uid', 'next', next);
	}

	/*
	 * Choose High
	*/
	high() {
		let { data, profit, round, next, result } = Queue.getOne('uid', this.id);

		if (round === 0) {
			profit = parseFloat(data.amount) / 3;
		}
		else {
			profit *= 2
		}
		round++;

		let currentCard = result[next];
		let nextCard = result[next + 1];

		let isWinner = false;

		if (currentCard >= nextCard) {
			isWinner = true;
		}

		Rule.checkLimited(this.id, this.data.coin, (status) => {
			if (!status) {
				result[next + 1] = currentCard - 1
				isWinner = false;
			}

			next++;
			this.emit({ command: 'high', target: result[next], winner: isWinner })
			Queue.update(this.id, 'uid', 'next', next);
			Queue.update(this.id, 'uid', 'profit', profit);
			Queue.update(this.id, 'uid', 'round', round);

			if (!isWinner) {
				this.selected = 'high';
				this.end(true);
			}
		})
	}

	/*
	 * Choose Low
	*/
	low() {
		let { data, profit, round, next, result } = Queue.getOne('uid', this.id);

		if (round === 0) {
			profit = parseFloat(data.amount) / 3;
		}
		else {
			profit *= 2
		}
		round++

		let currentCard = result[next];
		let nextCard = result[next + 1];

		let isWinner = false;

		if (currentCard <= nextCard) {
			isWinner = true;
		}

		Rule.checkLimited(this.id, this.data.coin, (status) => {
			if (!status) {
				result[next + 1] = currentCard + 1
				isWinner = false;
			}

			next++;
			this.emit({ command: 'low', target: result[next], winner: isWinner })
			Queue.update(this.id, 'uid', 'next', next);
			Queue.update(this.id, 'uid', 'profit', profit);
			Queue.update(this.id, 'uid', 'round', round);

			if (!isWinner) {
				this.selected = 'low';
				this.end(true);
			}
		})
	}

	/*
	 * End / Busted Game
	*/
	end(lost = false) {
		let self = this;

		let { cards, selected } = self.data;

		//Get The Client Game Info
		let info = Queue.getOne('uid', self.id);

		if (!info) {
			console.log('Client Not Playing!', self.id)
			return false;
		}

		if (_.isUndefined(cards)) {
			let r = info.result;
			r.length = 4;
			cards = r;
		}

		if (_.isUndefined(selected)) {
			selected = this.selected;
		}

		// let result = info.result;
		cards = _.flattenDeep(cards);

		// let last = _.last(cards);
		// var lastResult = result[cards.length];

		let result = { cards: cards, selected }

		let hash = info.hash;
		let profit = info.profit;
		let coin = info.coin;

		let isWinner = true;

		if (lost) {
			profit = -info.profit;
			isWinner = false;
		}

		profit = H.CryptoSet(profit, coin);

		Rule.prepareBusted(self.client, self.io, self.id, 'hilo', info.data, profit, result, hash, isWinner, (res) => {
			if (res) {
				self.emit({ command: 'busted', profit: profit });
				Queue.remove('uid', self.id)
			}
		})
	}
}

module.exports = Hilo;