const assert = require("assert");
const _ = require("lodash");
const Queue = require("../General/Queue");
const UserRule = require("../Users/Rule");
const H = require("../General/Helper");
const C = require("../General/Constant");
const Agent = require("./Agent");
const { encode, decode } = require("../General/Buffer");
const config = require("../config");

const houseEdge = config.house;

//Decelare Rule Object
const Rule = {};

/*
 * Check User Can Play Game
 * Add client to queue
 * When a client want to play a game, first client data can be check in here
 */
Rule.CanPlay = function (id, data, client, game, callback) {
  if (!id && !game) return;

  // If is Bot
  if (client === null) return callback(true);

  let { amount, coin } = data;

  if (!amount && !coin) return;

  id = _.toNumber(id);

  let bet = _.toString(amount);
  bet = _.replace(bet, ",", ".");
  bet = _.toNumber(bet);

  if (_.isNaN(bet)) {
    bet = parseFloat(bet);
  }

  if (bet === 0 && bet < 1e-7) {
    return callback(false);
  }

  if (!H.isValidNumber(bet)) {
    return callback(false);
  }

  // Security
  if (bet === "NaN" || bet === NaN) return callback(false);

  //Check BankRoll
  Agent.getBank(game, bet, coin, client, (can) => {
    if (!can) {
      return callback("Your Profit Can't be more than 5% of Bankroll !");
    }

    let exists = Queue.exists(id);

    if (game === "plinko") {
      exists = false;
    }

    // Check in Queue
    if (!exists) {
      // IMPORTANT SECURITY
      if (bet <= 0) return callback(false);
      if (bet === 0.0) return callback(false);
      if (bet <= 0.0) return callback(false);
      if (bet === "0.00000000") return callback(false);

      UserRule.getClientCoinCredit(id, coin, (result) => {
        if (result === false) {
          return callback("Your Balance is Not Enough", "credit");
        }

        let clientBalance = _.toNumber(result);

        if (clientBalance >= bet) {
          callback(true);
        } else {
          callback("Your Balance is Not Enough!", "credit");
        }
      });
    } else {
      Queue.remove("uid", id);

      //Check Again
      H.wait(150).then(() => {
        return Rule.CanPlay(id, data, client, game, callback);
      });
    }
  });
};

/*
 * Prepare Play Game
 * Add User to Queue
 * Reduce User Credit, Add BankRoll Amount
 */
Rule.preparePlay = function (client, uid, game, data, callback) {
  if (!uid && !game) return;

  let { amount, coin } = data;

  if (!amount && !coin) return;

  coin = _.lowerCase(coin);
  uid = _.toNumber(uid);

  //Make A New Bet Record
  UserRule.makeBetBeforePlay(data, uid, game, (gid, error) => {
    if (!error) {
      //Reduce Credit
      UserRule.reduceBalance(uid, amount, coin, (newBalance, err) => {
        if (err) {
          console.log("error on reduce balance from game rule: 92", err);
          return callback(false);
        }

        newBalance = H.CryptoSet(newBalance, coin);

        //Send Credit Change to Client
        if (client !== null)
          client.emit(
            C.UPDATE_CREDIT,
            encode({ coin: coin, value: newBalance })
          );

        UserRule.rakeAmount(uid, amount, coin);
        //Add Amount to BankRoll
        UserRule.addBankRoll(game, amount, coin, false, (status) => {
          // Add Client To Queue
          Queue.add({ uid, gid, game, coin, data });

          return callback(status);
        });
      });
    } else {
      console.log("error on game rule : 151", error);
    }
  });
};

/*
 * Busted Game / End Game
 * Remove User From Queue
 * Update User Credit, Update BankRoll Amount
 */
Rule.prepareBusted = function (
  client,
  io,
  uid,
  game,
  data,
  profit,
  result,
  hash,
  winner,
  callback
) {
  if (!uid && !game && !hash) return;

  uid = _.toNumber(uid);

  let queue = Queue.getOne("uid", uid);

  if (!queue) return callback(false, null);

  let gid = queue.gid;

  if (!gid) {
    return callback(false, null);
  }
  
  let { amount, coin } = data;

  if (_.isUndefined(amount)) amount = queue.data.amount;
  if (_.isUndefined(coin)) coin = queue.data.coin;

  if (!amount && !coin) return callback(false, null);

  coin = _.lowerCase(coin);
  amount = _.toNumber(amount);
  profit = _.toNumber(profit);

  //Calculate HouseEdge
  var percent = (houseEdge / 100) * H.CryptoSet(amount);
  var calculateHouse = H.CryptoSet(amount - percent);

  var amountAndProfit = profit + _.toNumber(calculateHouse);
  amountAndProfit = H.CryptoSet(amountAndProfit);

  //Update Bet Record
  UserRule.updateBetAfterFinish(
    gid,
    uid,
    profit,
    result,
    hash,
    (res, error) => {
      if (error) {
        console.log("Error on prepareBusted: 200");
        return callback(false, null);
      }

      // Add Client And Bet Details To Queue
      // Boardcast New Bet to All Clients
      UserRule.getNameById(uid, (name) => {
        if (!name) return callback(false, null);

        let data = {
          amount: H.CryptoSet(amount, coin),
          profit: H.CryptoSet(profit, coin),
          created: Date.now(),
          game: queue.game,
          result: result,
          direct: true,
          coin: coin,
          hash: hash,
          name: name,
          uid: uid,
          gid: gid,
        };

        io.emit(C.ADD_BET, encode(data));
      });

      // If is Bot
      if (client === null || _.isUndefined(client)) {
        //Update User Profit
        UserRule.updateProfit(true, true, uid, profit, coin, (ress) => {
          H.wait(10).then(() => {
            Queue.remove("uid", uid);
          });
          return callback(true, gid); // Return gid here
        });
      }

      //Client is Winner
      if (winner === true && profit !== 0) {
        //Add Credit
        UserRule.addBalance(uid, amountAndProfit, coin, (newBalance, err) => {
          if (err) return callback(false, null);

          //Send Credit Change to Client
          client.emit(
            C.UPDATE_CREDIT,
            encode({ coin: coin, value: newBalance })
          );

          //Reduce BankRoll
          UserRule.reduceBankRoll(game, amountAndProfit, coin, (status, er) => {
            if (er) {
              console.log("Error on reduceBankRoll");
              return callback(false, null);
            }
            if(coin === 'INR') {
              //Update Max Profit
              UserRule.updateMaxProfitinr(uid, profit, coin, (s) => {
                //Remove Client From Queue
                H.wait(10).then(() => {
                  Queue.remove("uid", uid);
                });
                //Return Status
                return callback(true, gid); // Return gid here
              });
            }
            else {
              //Update User Profit
              UserRule.updateProfit(false, true, uid, profit, coin, (isOk) => {
                if (isOk) {
                  //Update Max Profit
                  UserRule.updateMaxProfit(uid, profit, coin, (s) => {
                    //Remove Client From Queue
                    H.wait(10).then(() => {
                      Queue.remove("uid", uid);
                    });
                    //Return Status
                    return callback(true, gid); // Return gid here
                  });
                }
              });
            }
          });
        });
      }
      //Client is Lost
      else {
        //Update User Profit
        //Instead of profit, reduce amount from profit
        UserRule.updateProfit(false, false, uid, amount, coin, (isOk) => {
          if (isOk) {
            //Remove Client From Queue
            H.wait(10).then(() => {
              Queue.remove("uid", uid);
            });
            //Return Status
            return callback(true, gid); // Return gid here
          }
        });
      }
    }
  );
};

/*
 * Check Limited Profit
 */
Rule.checkLimited = function (id, coin, callback) {
  Agent.canProfit(id, coin, (status) => {
    callback(status);
  });
};

/*
 * Get The Last Profit [ NEW METHOD ]
 */
Rule.getLastProfit = function (id, callback) {
  UserRule.getUserLastProfit(id, (profit) => {
    if (profit) callback(profit.profit);
    else callback(0);
  });
};

//Export Rule Module
module.exports = Rule;
