/**
 * PostgresMiddleware.js
 * 
 * Handles postgres queries and caching to redis
 * 
 */
var _ = require('lodash')

var pg = require('pg'),
    crypto = require('crypto'),
    RedisMiddleware = require('./RedisMiddleware');

var { Client } = require('pg')
var types = require('pg').types;
var timestampOID = 1114;

types.setTypeParser(1114, function(stringValue) {
    return stringValue;
});

const config = require('../../config');

const client = new Client({
    connectionString: config.dbUrl,
});

client.connect();

////////////////////

/**
 * Execute a postgres query
 * @param {String}		query  	Querystring
 * @param {Array}		params 	Array of paramters for the query (empty array for none)
 * @param {Number}   	ttl    	Expiration of cache in seconds
 * @param {Function} 	callback   	callback in the form of function(err, data)
 */
function Query(query, params, cache, callback) {
    if (typeof params == 'function') {
        callback = params;
        params = [];
    }

    if (typeof cache == 'function') {
        callback = cache;
        cache = true;
    }

    // 1 Min
    var ttl = 1 * 60;

    // Return if no callback
    if (!callback) return console.log('YOU MUST PASS A CALL BACK TO QUERY FUNCTION!');

    // If we have caching enabled
    // Check and see if we have a cache in redis

    var hash = crypto.createHash('sha1').update(query + params.toString()).digest('hex');

    if(cache)
    {
        RedisMiddleware.get(hash, function(data) {
            if (!_.isNull(data)) {
                console.log('load from cache')
                return callback(false, data);
            } else {
                _execute(query, params, ttl, callback);
            }
        });
    }
    else {
        console.log('load from uncache')
        _execute(query, params, ttl, callback);
    }
}

function _execute(query, params, ttl, callback) {
    var hash = crypto.createHash('sha1').update(query + params.toString()).digest('hex');
    client.query(query, params, function(err, result) {
        if (err) {
            return console.log('err')
        } else {
            if (result === null || !result) {
                return console.log('ooops...')
            }
            RedisMiddleware.set(hash, ttl, result, function(data) {
                callback(false, result);
            });
        }
    });
}

module.exports = {
    query: Query
};