const db = require('../General/Model');

class AffiliateController {
  // Get Referral Code and Link for a User
  async getUserReferralInfo(userId) {
    try {
      const query = `
        SELECT "referalcode", "referral_link" 
        FROM users 
        WHERE id = $1
      `;
      const result = await db.query(query, [userId]);

      if (result.rows.length === 0) {
        throw new Error('User not found');
      }

      return {
        referralCode: result.rows[0].referalcode,
        referralLink: result.rows[0].referral_link
      };
    } catch (error) {
      console.error('Error fetching referral info:', error);
      throw error;
    }
  }

  // Get Team Data for a User by Referral Code
  async getTeamData(referralCode) {
    try {
      const teamQuery = `
        SELECT ownername, membername, "referalCode", "createdAt"
        FROM team
        WHERE "referalCode" = $1
      `;
      const teamResult = await db.query(teamQuery, [referralCode]);

      return {
        teamMembers: teamResult.rows
      };
    } catch (error) {
      console.error('Error fetching team data:', error);
      throw error;
    }
  }

  // Add Team Member
  async addTeamMember(ownername, username, referralCode) {
    try {
      const query = `
        INSERT INTO team(ownername, membername, "referalCode", "createdAt", "updatedAt") 
        VALUES($1, $2, $3, NOW(), NOW())
        RETURNING *
      `;
      const result = await db.query(query, [ownername, username, referralCode]);
      return result.rows[0];
    } catch (error) {
      console.error('Error adding team member:', error);
      throw error;
    }
  }

  // Record Affiliate Rewards
  async recordAffiliateReward(ownerName, memberName, referalCode, amount, coin = null) {
    try {
      const query = `
        INSERT INTO rewards(ownername, membername, "referalCode", amount, coin, "createdAt", "updatedAt")
        VALUES($1, $2, $3, $4, $5, NOW(), NOW())
        RETURNING *
      `;
      const result = await db.query(query, [ownerName, memberName, referalCode, amount, coin]);
      return result.rows[0];
    } catch (error) {
      console.error('Error recording affiliate reward:', error);
      throw error;
    }
  }

  // Get Rewards for a User
  async getUserRewards(referralCode) {
    try {
      const query = `
        SELECT ownername, membername, "referalCode", amount, coin,type,"createdAt",referalmount
        FROM rewards
        WHERE "referalCode" = $1
      `;
      const result = await db.query(query, [referralCode]);
      return {
        rewards: result.rows,
        totalRewards: result.rows.reduce((sum, reward) => sum + reward.amount, 0)
      };
    } catch (error) {
      console.error('Error fetching rewards:', error);
      throw error;
    }
  }
  
}

module.exports = new AffiliateController();


