const admin = require('../../firabsenotifcation/config');
const pg = require('../../General/Model');
const fs = require('fs');
const path = require('path');

class ClubBannerController {
    async createBanner(clubId, ownerId, title, imageFile) {
        try {
            await pg.query('BEGIN');

            // Get club details
            const clubQuery = 'SELECT unique_club_id FROM clubs WHERE id = $1 AND owner_id = $2';
            const clubResult = await pg.query(clubQuery, [clubId, ownerId]);
            
            if (clubResult.rows.length === 0) {
                throw new Error('Unauthorized: Not the club owner');
            }

            const uniqueClubId = clubResult.rows[0].unique_club_id;
            
            // Create directory if it doesn't exist
            const baseDir = '/var/www/html/hellogames/ibitplay/backend/clubmembership/clubbanners/images';
            const clubDir = path.join(baseDir, uniqueClubId);
            
            if (!fs.existsSync(clubDir)) {
                fs.mkdirSync(clubDir, { recursive: true });
            }

            // Save image with timestamp to ensure uniqueness
            const timestamp = Date.now();
            const imageName = `banner_${timestamp}${path.extname(imageFile.originalname)}`;
            const imagePath = path.join(clubDir, imageName);
            
            await fs.promises.writeFile(imagePath, imageFile.buffer);

            // Save banner record
            const insertQuery = `
                INSERT INTO club_banners (club_id, title, image_path)
                VALUES ($1, $2, $3)
                RETURNING *
            `;
            const result = await pg.query(insertQuery, [
                clubId,
                title,
                `${uniqueClubId}/${imageName}`
            ]);

            await pg.query('COMMIT');
            return result.rows[0];

        } catch (error) {
            await pg.query('ROLLBACK');
            throw error;
        }
    }

    async updateBanner(bannerId, clubId, ownerId, updates) {
        try {
            await pg.query('BEGIN');

            const bannerQuery = `
                SELECT cb.*, c.unique_club_id 
                FROM club_banners cb
                JOIN clubs c ON cb.club_id = c.id
                WHERE cb.id = $1 AND cb.club_id = $2 AND c.owner_id = $3
            `;
            const bannerResult = await pg.query(bannerQuery, [bannerId, clubId, ownerId]);
            
            if (bannerResult.rows.length === 0) {
                throw new Error('Banner not found or unauthorized');
            }

            const currentBanner = bannerResult.rows[0];
            
            let newImagePath = currentBanner.image_path;
            
            // Handle image update if provided
            if (updates.imageFile) {
                const baseDir = '/var/www/html/hellogames/ibitplay/backend/clubmembership/clubbanners/images';
                const timestamp = Date.now();
                const imageName = `banner_${timestamp}${path.extname(updates.imageFile.originalname)}`;
                const imagePath = path.join(baseDir, currentBanner.unique_club_id, imageName);
                
                await fs.promises.writeFile(imagePath, updates.imageFile.buffer);
                newImagePath = `${currentBanner.unique_club_id}/${imageName}`;

                // Delete old image
                const oldImagePath = path.join(baseDir, currentBanner.image_path);
                if (fs.existsSync(oldImagePath)) {
                    await fs.promises.unlink(oldImagePath);
                }
            }

            const updateQuery = `
                UPDATE club_banners
                SET title = COALESCE($1, title),
                    image_path = $2,
                    updated_at = CURRENT_TIMESTAMP
                WHERE id = $3
                RETURNING *
            `;
            
            const result = await pg.query(updateQuery, [
                updates.title || currentBanner.title,
                newImagePath,
                bannerId
            ]);

            await pg.query('COMMIT');
            return result.rows[0];

        } catch (error) {
            await pg.query('ROLLBACK');
            throw error;
        }
    }

    async deleteBanner(bannerId, clubId, ownerId) {
        try {
            await pg.query('BEGIN');

            const bannerQuery = `
                SELECT cb.*, c.unique_club_id 
                FROM club_banners cb
                JOIN clubs c ON cb.club_id = c.id
                WHERE cb.id = $1 AND cb.club_id = $2 AND c.owner_id = $3
            `;
            const bannerResult = await pg.query(bannerQuery, [bannerId, clubId, ownerId]);
            
            if (bannerResult.rows.length === 0) {
                throw new Error('Banner not found or unauthorized');
            }

            // Delete image file
            const imagePath = path.join(
                '/var/www/html/hellogames/ibitplay/backend/clubmembership/clubbanners/images',
                bannerResult.rows[0].image_path
            );
            
            if (fs.existsSync(imagePath)) {
                await fs.promises.unlink(imagePath);
            }

            // Delete banner record
            await pg.query('DELETE FROM club_banners WHERE id = $1', [bannerId]);

            await pg.query('COMMIT');
            return { success: true };

        } catch (error) {
            await pg.query('ROLLBACK');
            throw error;
        }
    }

    async getClubBanners(clubId) {
      const query = `
    SELECT 
        id,
        club_id,
        title,
        image_path,
        created_at,
        updated_at,
        is_active
    FROM club_banners
    WHERE club_id = $1 AND is_active = true
    ORDER BY created_at DESC
`;
        const result = await pg.query(query, [clubId]);
        return result.rows;
    }

    async sendBannerNotification(bannerId, clubId, ownerId, notification) {
        try {
            await pg.query('BEGIN');

            // Verify banner and ownership
            const bannerQuery = `
                SELECT cb.* FROM club_banners cb
                JOIN clubs c ON cb.club_id = c.id
                WHERE cb.id = $1 AND cb.club_id = $2 AND c.owner_id = $3
            `;
            const bannerResult = await pg.query(bannerQuery, [bannerId, clubId, ownerId]);
            
            if (bannerResult.rows.length === 0) {
                throw new Error('Banner not found or unauthorized');
            }

            // Insert notification
            const notifQuery = `
                INSERT INTO club_banner_notifications 
                (banner_id, club_id, sender_id, title, body, type, additional_data)
                VALUES ($1, $2, $3, $4, $5, $6, $7)
                RETURNING id
            `;
            const notifResult = await pg.query(notifQuery, [
                bannerId,
                clubId,
                ownerId,
                notification.title,
                notification.body,
                'banner_promotion',
                { ...notification.data, bannerId: bannerId }
            ]);
            
            const notificationId = notifResult.rows[0].id;

            // Get all club members
            const membersQuery = `
                SELECT 
                    cm.user_id,
                    uft.token,
                    uft.is_active
                FROM club_memberships cm
                LEFT JOIN user_fcm_tokens uft ON cm.user_id = uft.user_id
                WHERE cm.club_id = $1
            `;
            const members = await pg.query(membersQuery, [clubId]);

            // Create notification status records
            const statusPromises = members.rows.map(member => {
                return pg.query(
                    `INSERT INTO club_banner_notification_status 
                    (notification_id, user_id, is_sent, sent_at)
                    VALUES ($1, $2, $3, $4)`,
                    [
                        notificationId,
                        member.user_id,
                        member.token ? true : false,
                        member.token ? 'NOW()' : null
                    ]
                );
            });

            await Promise.all(statusPromises);

            // Send FCM notifications
            const validMembers = members.rows.filter(m => m.token && m.is_active);
            
            if (validMembers.length > 0) {
                const messages = validMembers.map(member => ({
                    token: member.token,
                    notification: {
                        title: notification.title,
                        body: notification.body
                    },
                    data: {
                        ...notification.data,
                        notificationId: notificationId.toString(),
                        type: 'banner_promotion',
                        bannerId: bannerId.toString(),
                        clubId: clubId.toString()
                    }
                }));

                await Promise.allSettled(messages.map(msg => admin.messaging().send(msg)));
            }

            await pg.query('COMMIT');
            
            return {
                notificationId,
                totalMembers: members.rows.length,
                sentCount: validMembers.length
            };

        } catch (error) {
            await pg.query('ROLLBACK');
            throw error;
        }
    }
}

module.exports = new ClubBannerController();