// fileupload.js
const multer = require('multer');
const fs = require('fs');
const path = require('path');

// Define base upload folder
const baseUploadFolder = '/var/www/html/hellogames/ibitplay/backend/clubmembership/clubprofile';

// Create storage configuration
const storage = multer.diskStorage({
  destination: (req, file, cb) => {
    try {
      // Generate a temporary upload directory using timestamp
      const timestamp = Date.now();
      const tempUploadDir = path.join(baseUploadFolder, `temp_${timestamp}`);

      // Create directory if it doesn't exist
      if (!fs.existsSync(tempUploadDir)) {
        fs.mkdirSync(tempUploadDir, { recursive: true });
      }

      // Store the temp directory path in the request for later use
      req.tempUploadDir = tempUploadDir;
      
      cb(null, tempUploadDir);
    } catch (error) {
      cb(error);
    }
  },
  filename: (req, file, cb) => {
    const uniqueSuffix = Date.now() + '-' + Math.round(Math.random() * 1E9);
    const extension = path.extname(file.originalname);
    const filename = `profile-${uniqueSuffix}${extension}`;
    
    // Store the filename in the request for later use
    req.profileFilename = filename;
    
    cb(null, filename);
  }
});

// File filter configuration
const fileFilter = (req, file, cb) => {
  const allowedMimeTypes = ['image/jpeg', 'image/png', 'image/gif'];
  if (allowedMimeTypes.includes(file.mimetype)) {
    cb(null, true);
  } else {
    cb(new Error('Only .jpg, .png and .gif files are allowed'));
  }
};

// File size limits
const limits = {
  fileSize: 5 * 1024 * 1024, // 5MB limit
  files: 1 // Only 1 file at a time
};

// Create and export the multer instance directly
const upload = multer({
  storage,
  fileFilter,
  limits
});

// Helper function to move file to final destination
const moveFileToClubFolder = async (tempDir, clubName, filename) => {
  const sanitizedClubName = clubName.replace(/[^a-zA-Z0-9]/g, '_').toLowerCase();
  const finalDir = path.join(baseUploadFolder, sanitizedClubName);
  
  try {
    // Create final directory if it doesn't exist
    if (!fs.existsSync(finalDir)) {
      fs.mkdirSync(finalDir, { recursive: true });
    }

    // Move file from temp to final location
    const oldPath = path.join(tempDir, filename);
    const newPath = path.join(finalDir, filename);
    fs.renameSync(oldPath, newPath);

    // Remove temporary directory
    fs.rmdirSync(tempDir, { recursive: true });

    return filename;
  } catch (error) {
    throw new Error(`Failed to move file: ${error.message}`);
  }
};

module.exports = {
  upload,
  moveFileToClubFolder
};