const express = require('express');
const router = express.Router();
const NotificationService = require('./controller');

router.post('/register', async (req, res) => {
    try {
        const { userId, token, deviceType, platform } = req.body;
        const result = await NotificationService.registerToken(
            userId, token, deviceType, platform
        );
        res.json({ success: true, data: result });
    } catch (error) {
        res.status(500).json({ success: false, error: error.message });
    }
});

// router.post('/send-to-user', async (req, res) => {
//     try {
//         const { userId, title, body, type, data } = req.body;
//         const notifId = await NotificationService.sendToUser(
//             userId, { title, body, type, data }
//         );
//         res.json({ success: true, notificationId: notifId });
//     } catch (error) {
//         res.status(500).json({ success: false, error: error.message });
//     }
// });
router.post('/send-to-user', async (req, res) => {
    console.log('[NotificationAPI] Received request:', {
        userId: req.body.userId,
        title: req.body.title,
        type: req.body.type,
        dataKeys: req.body.data ? Object.keys(req.body.data) : []
    });

    try {
        const { userId, title, body, type, data } = req.body;
        
        console.log('[NotificationAPI] Validating request parameters');
        if (!userId || !title || !body) {
            console.warn('[NotificationAPI] Missing required parameters:', { userId, title, body });
            return res.status(400).json({ 
                success: false, 
                error: 'Missing required parameters'
            });
        }

        console.log('[NotificationAPI] Calling NotificationService.sendToUser');
        const notifId = await NotificationService.sendToUser(
            userId, { title, body, type, data }
        );
        
        console.log('[NotificationAPI] Successfully sent notification:', {
            notificationId: notifId,
            userId
        });
        
        res.json({ success: true, notificationId: notifId });
    } catch (error) {
        console.error('[NotificationAPI] Error in send-to-user route:', {
            error: {
                name: error.name,
                message: error.message,
                stack: error.stack
            },
            requestBody: {
                userId: req.body.userId,
                type: req.body.type,
                hasData: !!req.body.data
            }
        });
        
        res.status(500).json({ 
            success: false, 
            error: error.message
        });
    }
});
router.post('/send-bulk', async (req, res) => {
    try {
        const { title, body, type, data } = req.body;
        const results = await NotificationService.sendBulk(
            { title, body, type, data }
        );
        res.json({ success: true, results });
    } catch (error) {
        res.status(500).json({ success: false, error: error.message });
    }
});

router.get('/unread-count/:userId', async (req, res) => {
    try {
        const { userId } = req.params;
        const count = await NotificationService.getUnreadCount(userId);
        res.json({ success: true, unreadCount: count });
    } catch (error) {
        res.status(500).json({ success: false, error: error.message });
    }
});

router.post('/mark-as-read', async (req, res) => {
    try {
        const { userId, notificationIds } = req.body;
        if (!Array.isArray(notificationIds)) {
            throw new Error('notificationIds must be an array');
        }
        const updatedNotifications = await NotificationService.markAsRead(
            userId, 
            notificationIds
        );
        res.json({ 
            success: true, 
            updatedNotifications 
        });
    } catch (error) {
        res.status(500).json({ success: false, error: error.message });
    }
});

router.get('/history/:userId', async (req, res) => {
    try {
        const { userId } = req.params;
        const history = await NotificationService.getNotificationHistory(userId);
        res.json({ 
            success: true, 
            notifications: history 
        });
    } catch (error) {
        res.status(500).json({ success: false, error: error.message });
    }
});

router.get('/allToken', async (req, res) => {
    try {
        const tokens = await NotificationService.getAllTokens();
        res.json({ 
            success: true, 
            tokens: tokens
        });
    } catch (error) {
        res.status(500).json({ success: false, error: error.message });
    }
});

module.exports = router;