// controllers/swapController.js

const pg = require('../General/Model');

const swapController = {
    // Get user's available balances
    getUserBalances: async (req, res) => {
        try {
            const { uid } = req.params;
           const query = `
    SELECT 
        uid,
        btc,
        eth,
        ltc,
        bch,
        usdt,
        trx,
        doge,
        ada,
        xrp,
        bnb,
        usdp,
        nexo,
        mkr,
        tusd,
        usdc,
        busd,
        nc,
        inr,
        shib,
        matic,
        sc,
        mvr,
        bjb,
        aed,
        npr,
        pkr
    FROM credits 
    WHERE uid = $1
`;
            const result = await pg.query(query, [uid]);

            if (result.rows.length === 0) {
                return res.status(404).json({
                    success: false,
                    message: 'User not found'
                });
            }

            res.status(200).json({
                success: true,
                data: result.rows[0]
            });
        } catch (error) {
            res.status(500).json({
                success: false,
                message: 'Error fetching user balances',
                error: error.message
            });
        }
    },

    // Perform currency swap
   
    swapCurrency: async (req, res) => {
        try {
            const { uid, fromCurrency, toCurrency, amount } = req.body;
            console.log(req.body);
            
            // Validation
            if (!uid || !fromCurrency || !toCurrency || !amount) {
                return res.status(400).json({
                    success: false,
                    message: 'Missing required parameters'
                });
            }
    
            if (fromCurrency === toCurrency) {
                return res.status(400).json({
                    success: false,
                    message: 'Cannot swap to same currency'
                });
            }
    
            if (amount <= 0) {
                return res.status(400).json({
                    success: false,
                    message: 'Amount must be greater than 0'
                });
            }
    
            await pg.query('BEGIN');
    
            // Get exchange rates
            const ratesQuery = `
                SELECT currency, usd_rate 
                FROM exchangerate 
                WHERE UPPER(currency) IN (UPPER($1), UPPER($2))
            `;
            const ratesResult = await pg.query(ratesQuery, [fromCurrency, toCurrency]);
    
            if (ratesResult.rows.length !== 2) {
                await pg.query('ROLLBACK');
                return res.status(404).json({
                    success: false,
                    message: 'One or both currencies not found in exchange rates'
                });
            }
    
            const rates = ratesResult.rows.reduce((acc, curr) => {
                acc[curr.currency.toLowerCase()] = curr.usd_rate;
                return acc;
            }, {});
    
            // Get user's current balance
            const balanceQuery = `
                SELECT ${fromCurrency.toLowerCase()} as from_balance, 
                       ${toCurrency.toLowerCase()} as to_balance 
                FROM credits 
                WHERE uid = $1
            `;
            const balanceResult = await pg.query(balanceQuery, [uid]);
    
            if (balanceResult.rows.length === 0) {
                await pg.query('ROLLBACK');
                return res.status(404).json({
                    success: false,
                    message: 'User not found'
                });
            }
    
            const userBalance = balanceResult.rows[0];
    
            if (userBalance.from_balance < amount) {
                await pg.query('ROLLBACK');
                return res.status(400).json({
                    success: false,
                    message: 'Insufficient balance'
                });
            }
    
            // Calculate fee (1% for non-BJT currencies)
            const feePercentage = fromCurrency.toLowerCase() === 'bjt' ? 0  :fromCurrency.toLowerCase() === 'inr' ? 0.15  : 0.01;
            const feeAmount = amount * feePercentage;
            const amountAfterFee = amount - feeAmount;
    
            // Calculate conversion
            const usdAmount = amountAfterFee * rates[fromCurrency.toLowerCase()];
            const convertedAmount = usdAmount / rates[toCurrency.toLowerCase()];
    
            // Update balances
            const updateQuery = `
                UPDATE credits 
                SET ${fromCurrency.toLowerCase()} = ${fromCurrency.toLowerCase()} - $1,
                    ${toCurrency.toLowerCase()} = ${toCurrency.toLowerCase()} + $2
                WHERE uid = $3
                RETURNING ${fromCurrency.toLowerCase()} as new_from_balance,
                          ${toCurrency.toLowerCase()} as new_to_balance
            `;
            const updateResult = await pg.query(updateQuery, [amount, convertedAmount, uid]);
    
            // Create swap history record
            const historyQuery = `
                INSERT INTO swap_history (
                    uid,
                    from_currency,
                    to_currency,
                    from_amount,
                    to_amount,
                    fee_percentage,
                    fee_amount,
                    fee_currency,
                    usd_rate_from,
                    usd_rate_to,
                    created_at
                ) VALUES ($1, $2, $3, $4, $5, $6, $7, $8, $9, $10, NOW())
            `;
            
            await pg.query(historyQuery, [
                uid,
                fromCurrency,
                toCurrency,
                amount,
                convertedAmount,
                feePercentage,
                feeAmount,
                fromCurrency,
                rates[fromCurrency.toLowerCase()],
                rates[toCurrency.toLowerCase()]
            ]);
    
            await pg.query('COMMIT');
    
            res.status(200).json({
                success: true,
                message: 'Swap completed successfully',
                data: {
                    fromCurrency,
                    fromAmount: amount,
                    feePercentage: feePercentage * 100,
                    feeAmount,
                    amountAfterFee,
                    toCurrency,
                    toAmount: convertedAmount,
                    newBalances: {
                        [fromCurrency]: updateResult.rows[0].new_from_balance,
                        [toCurrency]: updateResult.rows[0].new_to_balance
                    }
                }
            });
    
        } catch (error) {
            await pg.query('ROLLBACK');
            res.status(500).json({
                success: false,
                message: 'Error performing swap',
                error: error.message
            });
        } finally {
            console.log("finished swap");
        }
    },
    // Get estimated swap amount without performing the swap
    getSwapEstimate: async (req, res) => {
        try {
            const { fromCurrency, toCurrency, amount } = req.query;

            if (!fromCurrency || !toCurrency || !amount) {
                return res.status(400).json({
                    success: false,
                    message: 'Missing required parameters'
                });
            }

            const ratesQuery = `
                SELECT currency, usd_rate 
                FROM exchangerate 
                WHERE UPPER(currency) IN (UPPER($1), UPPER($2))
            `;
            const ratesResult = await pg.query(ratesQuery, [fromCurrency, toCurrency]);

            if (ratesResult.rows.length !== 2) {
                return res.status(404).json({
                    success: false,
                    message: 'One or both currencies not found in exchange rates'
                });
            }

            const rates = ratesResult.rows.reduce((acc, curr) => {
                acc[curr.currency.toLowerCase()] = curr.usd_rate;
                return acc;
            }, {});

            const usdAmount = amount * rates[fromCurrency.toLowerCase()];
            const estimatedAmount = usdAmount / rates[toCurrency.toLowerCase()];

            res.status(200).json({
                success: true,
                data: {
                    fromCurrency,
                    fromAmount: parseFloat(amount),
                    toCurrency,
                    estimatedAmount,
                    exchangeRates: {
                        [fromCurrency]: rates[fromCurrency.toLowerCase()],
                        [toCurrency]: rates[toCurrency.toLowerCase()]
                    }
                }
            });

        } catch (error) {
            res.status(500).json({
                success: false,
                message: 'Error calculating swap estimate',
                error: error.message
            });
        }
    },
    getAllSwapHistory: async (req, res) => {
        try {
            const query = `
                SELECT 
                    sh.*,
                    u.name as user_name
                FROM swap_history sh
                LEFT JOIN users u ON sh.uid = u.id
                ORDER BY sh.created_at DESC
            `;
            
            const result = await pg.query(query);

            res.status(200).json({
                success: true,
                data: result.rows
            });
        } catch (error) {
            res.status(500).json({
                success: false,
                message: 'Error fetching swap history',
                error: error.message
            });
        }
    },

    // Get swap history for specific user
    getUserSwapHistory: async (req, res) => {
        try {
            const { uid } = req.params;

            // Validate uid
            if (!uid) {
                return res.status(400).json({
                    success: false,
                    message: 'User ID is required'
                });
            }

            const query = `
                SELECT 
                    sh.*,
                    u.name as user_name
                FROM swap_history sh
                LEFT JOIN users u ON sh.uid = u.id
                WHERE sh.uid = $1
                ORDER BY sh.created_at DESC
            `;
            
            const result = await pg.query(query, [uid]);

            if (result.rows.length === 0) {
                return res.status(404).json({
                    success: false,
                    message: 'No swap history found for this user'
                });
            }

            res.status(200).json({
                success: true,
                data: result.rows
            });
        } catch (error) {
            res.status(500).json({
                success: false,
                message: 'Error fetching user swap history',
                error: error.message
            });
        }
    }
};

module.exports = swapController;