const pg = require('../General/Model');

 const getUserReports = async (req, res) => {
  try {
    const { userId } = req.params;
    
    if (!userId) {
      return res.status(400).json({ error: 'User ID is required' });
    }

    // Get user details
    const userQuery = `
      SELECT id, name, avatar, level, games_played, referalcode 
      FROM users 
      WHERE id = $1
    `;
    
    // Get user balance from credits table
   const creditsQuery = `
    SELECT 
        uid,
        btc,
        eth,
        ltc,
        bch,
        usdt,
        trx,
        doge,
        ada,
        xrp,
        bnb,
        usdp,
        nexo,
        mkr,
        tusd,
        usdc,
        busd,
        nc,
        inr,
        shib,
        matic,
        sc,
        mvr,
        bjb,
        aed,
        npr,
        pkr
    FROM credits 
    WHERE uid = $1
`;
    
    // Get user wager
    const wagerQuery = `
      SELECT wager FROM userwager 
      WHERE uid = $1
    `;

    const [userResult, creditsResult, wagerResult] = await Promise.all([
      pg.query(userQuery, [userId]),
      pg.query(creditsQuery, [userId]),
      pg.query(wagerQuery, [userId])
    ]);

    const user = userResult.rows[0] || {};
    const credits = creditsResult.rows[0] || {};
    const wagerString = wagerResult.rows[0]?.wager || "0"; // Get wager as string
    const wager = parseFloat(wagerString.replace(/,/g, ""));

    // Calculate VIP level details
    const vipDetails = getVipLevelDetails(wager);

    return res.status(200).json({
      user,
      credits,
      wager,
      vipDetails
    });
  } catch (error) {
    console.error('Error fetching user reports:', error);
    return res.status(500).json({ error: 'Failed to fetch user reports' });
  }
};


const getAllUsersReports = async (req, res) => {
  try {
    const { page = 1, limit = 10, search = '' } = req.query;
    const offset = (Number(page) - 1) * Number(limit);
    
    let userQuery = `
      SELECT u.id, u.name, u.avatar, u.level, u.games_played, u.referalcode
      FROM users u
    `;
    
    let countQuery = `SELECT COUNT(*) FROM users u`;
    
    let params = [];
    let paramCounter = 1;
    
    // Add search condition if provided
    if (search && search.trim() !== '') {
      const searchTerm = `%${search.trim()}%`;
      const searchCondition = `WHERE u.name ILIKE $${paramCounter} OR u.referalcode ILIKE $${paramCounter}`;
      userQuery += ` ${searchCondition}`;
      countQuery += ` ${searchCondition}`;
      params.push(searchTerm);
      paramCounter++;
    }
    
    // Add pagination with correct parameter indices
    userQuery += ` ORDER BY u.id DESC LIMIT $${paramCounter} OFFSET $${paramCounter + 1}`;
    params.push(Number(limit), Number(offset));
    
    // Execute queries
    const [usersResult, countResult] = await Promise.all([
      pg.query(userQuery, params),
      pg.query(countQuery, params.slice(0, search && search.trim() !== '' ? 1 : 0))
    ]);
    
    const users = usersResult.rows;
    const totalUsers = parseInt(countResult.rows[0].count);
    
    // Get additional data for each user
    const usersWithData = await Promise.all(
      users.map(async (user) => {
        // Get credits
      const creditsQuery = `SELECT 
    uid,
    btc,
    eth,
    ltc,
    bch,
    usdt,
    trx,
    doge,
    ada,
    xrp,
    bnb,
    usdp,
    nexo,
    mkr,
    tusd,
    usdc,
    busd,
    nc,
    inr,
    shib,
    matic,
    sc,
    mvr,
    bjb,
    aed,
    npr,
    pkr
FROM credits WHERE uid = $1`;
        const creditsResult = await pg.query(creditsQuery, [user.id]);
        const credits = creditsResult.rows[0] || {};
        
        // Get wager
        const wagerQuery = `SELECT wager FROM userwager WHERE uid = $1`;
        const wagerResult = await pg.query(wagerQuery, [user.id]);
        const wagerString = wagerResult.rows[0]?.wager || "0";
        const wager = parseFloat(wagerString.replace(/,/g, ""));
        
        // Calculate VIP level
        const vipDetails = getVipLevelDetails(wager);
        
        return {
          ...user,
          credits,
          wager,
          vipDetails
        };
      })
    );
    
    return res.status(200).json({
      users: usersWithData,
      pagination: {
        total: totalUsers,
        page: Number(page),
        limit: Number(limit),
        pages: Math.ceil(totalUsers / Number(limit))
      }
    });
  } catch (error) {
    console.error('Error fetching all users reports:', error);
    return res.status(500).json({ error: 'Failed to fetch users reports' });
  }
};
// VIP level calculation function
function getVipLevelDetails(xpAmount) {
    // VIP levels array
    const vipLevels = [
      { level: 'VIP 01', minXP: 1, maxXP: 99, card: 'brownz', iconPath: '/assets/images/starVip.png' },
      { level: 'VIP 02', minXP: 100, maxXP: 199, card: 'brownz', iconPath: '/assets/images/starVip.png' },
      { level: 'VIP 03', minXP: 200, maxXP: 999, card: 'brownz', iconPath: '/assets/images/starVip.png' },
      { level: 'VIP 04', minXP: 1000, maxXP: 1999, card: 'brownz', iconPath: '/assets/images/starVip.png' },
      { level: 'VIP 05', minXP: 2000, maxXP: 2999, card: 'brownz', iconPath: '/assets/images/starVip.png' },
      { level: 'VIP 06', minXP: 3000, maxXP: 3999, card: 'brownz', iconPath: '/assets/images/starVip.png' },
      { level: 'VIP 07', minXP: 4000, maxXP: 4999, card: 'brownz', iconPath: '/assets/images/starVip.png' },
      { level: 'VIP 08', minXP: 5000, maxXP: 6999, card: 'silver', iconPath: '/assets/images/starVip.png' },
      { level: 'VIP 09', minXP: 7000, maxXP: 8999, card: 'silver', iconPath: '/assets/images/starVip.png' },
      { level: 'VIP 10', minXP: 9000, maxXP: 10999, card: 'silver', iconPath: '/assets/images/starVip.png' },
      { level: 'VIP 11', minXP: 11000, maxXP: 12999, card: 'silver', iconPath: '/assets/images/starVip.png' },
      { level: 'VIP 12', minXP: 13000, maxXP: 14999, card: 'silver', iconPath: '/assets/images/starVip.png' },
      { level: 'VIP 13', minXP: 15000, maxXP: 16999, card: 'silver', iconPath: '/assets/images/starVip.png' },
      { level: 'VIP 14', minXP: 17000, maxXP: 18999, card: 'silver', iconPath: '/assets/images/starVip.png' },
      { level: 'VIP 15', minXP: 19000, maxXP: 20999, card: 'silver', iconPath: '/assets/images/starVip.png' },
      { level: 'VIP 16', minXP: 21000, maxXP: 22999, card: 'silver', iconPath: '/assets/images/starVip.png' },
      { level: 'VIP 17', minXP: 23000, maxXP: 24999, card: 'silver', iconPath: '/assets/images/starVip.png' },
      { level: 'VIP 18', minXP: 25000, maxXP: 26999, card: 'silver', iconPath: '/assets/images/starVip.png' },
      { level: 'VIP 19', minXP: 27000, maxXP: 28999, card: 'silver', iconPath: '/assets/images/starVip.png' },
      { level: 'VIP 20', minXP: 29000, maxXP: 30999, card: 'silver', iconPath: '/assets/images/starVip.png' },
      { level: 'VIP 21', minXP: 31000, maxXP: 44999, card: 'silver', iconPath: '/assets/images/starVip.png' },
      { level: 'VIP 22', minXP: 45000, maxXP: 48999, card: 'gold', iconPath: '/assets/images/goldStar.png' },
      { level: 'VIP 23', minXP: 49000, maxXP: 58999, card: 'gold', iconPath: '/assets/images/goldStar.png' },
      { level: 'VIP 24', minXP: 59000, maxXP: 68999, card: 'gold', iconPath: '/assets/images/goldStar.png' },
      { level: 'VIP 25', minXP: 69000, maxXP: 78999, card: 'gold', iconPath: '/assets/images/goldStar.png' },
      { level: 'VIP 26', minXP: 79000, maxXP: 88999, card: 'gold', iconPath: '/assets/images/goldStar.png' },
      { level: 'VIP 27', minXP: 89000, maxXP: 98999, card: 'gold', iconPath: '/assets/images/goldStar.png' },
      { level: 'VIP 28', minXP: 99000, maxXP: 108999, card: 'gold', iconPath: '/assets/images/goldStar.png' },
      { level: 'VIP 29', minXP: 109000, maxXP: 118999, card: 'gold', iconPath: '/assets/images/goldStar.png' },
      { level: 'VIP 30', minXP: 119000, maxXP: 128999, card: 'gold', iconPath: '/assets/images/goldStar.png' },
      { level: 'VIP 31', minXP: 129000, maxXP: 138999, card: 'gold', iconPath: '/assets/images/goldStar.png' },
      { level: 'VIP 32', minXP: 139000, maxXP: 148999, card: 'gold', iconPath: '/assets/images/goldStar.png' },
      { level: 'VIP 33', minXP: 149000, maxXP: 158999, card: 'gold', iconPath: '/assets/images/goldStar.png' },
      { level: 'VIP 34', minXP: 159000, maxXP: 168999, card: 'gold', iconPath: '/assets/images/goldStar.png' },
      { level: 'VIP 35', minXP: 169000, maxXP: 178999, card: 'gold', iconPath: '/assets/images/goldStar.png' },
      { level: 'VIP 36', minXP: 179000, maxXP: 188999, card: 'gold', iconPath: '/assets/images/goldStar.png' },
      { level: 'VIP 37', minXP: 189000, maxXP: 296999, card: 'gold', iconPath: '/assets/images/goldStar.png' },
      { level: 'VIP 38', minXP: 297000, maxXP: 320999, card: 'platinum', iconPath: '/assets/images/platinumStar.webp' },
      { level: 'VIP 39', minXP: 321000, maxXP: 376999, card: 'platinum', iconPath: '/assets/images/platinumStar.webp' },
      { level: 'VIP 40', minXP: 377000, maxXP: 432999, card: 'platinum', iconPath: '/assets/images/platinumStar.webp' },
      { level: 'VIP 41', minXP: 433000, maxXP: 488999, card: 'platinum', iconPath: '/assets/images/platinumStar.webp' },
      { level: 'VIP 42', minXP: 489000, maxXP: 544999, card: 'platinum', iconPath: '/assets/images/platinumStar.webp' },
      { level: 'VIP 43', minXP: 545000, maxXP: 600999, card: 'platinum', iconPath: '/assets/images/platinumStar.webp' },
      { level: 'VIP 44', minXP: 601000, maxXP: 656999, card: 'platinum', iconPath: '/assets/images/platinumStar.webp' },
      { level: 'VIP 45', minXP: 657000, maxXP: 712999, card: 'platinum', iconPath: '/assets/images/platinumStar.webp' },
      { level: 'VIP 46', minXP: 713000, maxXP: 768999, card: 'platinum', iconPath: '/assets/images/platinumStar.webp' },
      { level: 'VIP 47', minXP: 769000, maxXP: 824999, card: 'platinum', iconPath: '/assets/images/platinumStar.webp' },
      { level: 'VIP 48', minXP: 825000, maxXP: 880999, card: 'platinum', iconPath: '/assets/images/platinumStar.webp' },
      { level: 'VIP 49', minXP: 881000, maxXP: 936999, card: 'platinum', iconPath: '/assets/images/platinumStar.webp' },
      { level: 'VIP 50', minXP: 937000, maxXP: 992999, card: 'platinum', iconPath: '/assets/images/platinumStar.webp' },
      { level: 'VIP 51', minXP: 993000, maxXP: 1048999, card: 'platinum', iconPath: '/assets/images/platinumStar.webp' },
      { level: 'VIP 52', minXP: 1049000, maxXP: 1104999, card: 'platinum', iconPath: '/assets/images/platinumStar.webp' },
      { level: 'VIP 53', minXP: 1105000, maxXP: 1160999, card: 'platinum', iconPath: '/assets/images/platinumStar.webp' },
      { level: 'VIP 54', minXP: 1161000, maxXP: 1216999, card: 'platinum', iconPath: '/assets/images/platinumStar.webp' },
      { level: 'VIP 55', minXP: 1217000, maxXP: 1272999, card: 'platinum', iconPath: '/assets/images/platinumStar.webp' },
      { level: 'VIP 56', minXP: 1273000, maxXP: 2368999, card: 'platinum', iconPath: '/assets/images/platinumStar.webp' },
      { level: 'VIP 57', minXP: 2369000, maxXP: 2656999, card: 'platinum', iconPath: '/assets/images/platinumStar.webp' },
      { level: 'VIP 58', minXP: 2657000, maxXP: 2944999, card: 'platinum', iconPath: '/assets/images/platinumStar.webp' },
      { level: 'VIP 59', minXP: 2945000, maxXP: 3232999, card: 'platinum', iconPath: '/assets/images/platinumStar.webp' },
      { level: 'VIP 60', minXP: 3233000, maxXP: 3520999, card: 'platinum', iconPath: '/assets/images/platinumStar.webp' },
      { level: 'VIP 61', minXP: 3521000, maxXP: 3808999, card: 'platinum', iconPath: '/assets/images/platinumStar.webp' },
      { level: 'VIP 62', minXP: 3809000, maxXP: 4096999, card: 'platinum', iconPath: '/assets/images/platinumStar.webp' },
      { level: 'VIP 63', minXP: 4097000, maxXP: 4384999, card: 'platinum', iconPath: '/assets/images/platinumStar.webp' },
      { level: 'VIP 64', minXP: 4385000, maxXP: 4672999, card: 'platinum', iconPath: '/assets/images/platinumStarPandina/d1.webp' },
      { level: 'VIP 65', minXP: 4673000, maxXP: 4960999, card: 'platinum', iconPath: '/assets/images/platinumStar.webp' },
      { level: 'VIP 66', minXP: 4961000, maxXP: 5249999, card: 'platinum', iconPath: '/assets/images/platinumStar.webp' },
      { level: 'VIP 67', minXP: 5250000, maxXP: 5537999, card: 'platinum', iconPath: '/assets/images/platinumStar.webp' },
      { level: 'VIP 68', minXP: 5538000, maxXP: 8576999, card: 'platinum', iconPath: '/assets/images/platinumStar.webp' },
      { level: 'VIP 69', minXP: 8577000, maxXP: 9216999, card: 'platinum', iconPath: '/assets/images/platinumStar.webp' },
      { level: 'VIP 70', minXP: 9217000, maxXP: 10872832999, card: 'diamond', iconPath: '/assets/images/bcgames/vip/d1.webp' },
      { level: 'VIP 71', minXP: 10872833000, maxXP: 12058624999, card: 'diamond', iconPath: '/assets/images/bcgames/vip/d1.webp' },
      { level: 'VIP 72', minXP: 12058625000, maxXP: 13058624999, card: 'diamond', iconPath: '/assets/images/bcgames/vip/d1.webp' },
      { level: 'VIP 73', minXP: 13058625000, maxXP: 14058624999, card: 'diamond', iconPath: '/assets/images/bcgames/vip/d1.webp' },
      { level: 'VIP 74', minXP: 14058625000, maxXP: 15058624999, card: 'diamond', iconPath: '/assets/images/bcgames/vip/d1.webp' },
      { level: 'VIP 75', minXP: 15058625000, maxXP: 99999999999, card: 'diamond', iconPath: '/assets/images/bcgames/vip/d1.webp' }
  ];

    // Parse xpAmount to float to ensure correct calculations
    xpAmount = parseFloat(xpAmount);

    // Handle base case when xp is 0
    if (xpAmount === 0) {
        return {
            vipLevel: 'VIP 00',
            currentWagger: '0.0000',
            waggerNeededForNextLevel: '1.0000',
            completionPercentage: '0.00',
            card: 'brownz'
        };
    }

    // Find the correct vip level based on xp amount
    for (let i = 0; i < vipLevels.length; i++) {
        const vip = vipLevels[i];
        
        if (xpAmount >= vip.minXP && xpAmount <= vip.maxXP) {
            const nextVip = vipLevels[i + 1] || vip; 
            const totalXPForLevel = vip.maxXP - vip.minXP + 1; 
            const xpInCurrentLevel = xpAmount - vip.minXP; 
            const completionPercentage = ((xpInCurrentLevel / totalXPForLevel) * 100).toFixed(2); 
            
            return {
                vipLevel: nextVip.level,
                previousVipLevel: vip.level,
                currentWagger: xpAmount.toFixed(2), 
                waggerNeededForNextLevel: (vip.maxXP - xpAmount).toFixed(2), 
                completionPercentage: completionPercentage, 
                card: vip.card
            };
        }
    }

    // If xpAmount exceeds the maximum VIP level
    return {
        error: "XP amount exceeds the maximum VIP level"
    };
}

const exportUsersReports = async (req, res) => {
  try {
    const { search = '' } = req.query;
    
    // Base query to get all users
    let userQuery = `
      SELECT u.id, u.name, u.avatar, u.level, u.games_played, u.referalcode
      FROM users u
    `;

    // Add search condition if provided
    if (search) {
      userQuery += ` WHERE u.name ILIKE $1 OR u.referalcode ILIKE $1`;
    }

    // Execute query
    const params = search ? [`%${search}%`] : [];
    const usersResult = await pg.query(userQuery, params);
    const users = usersResult.rows;

    // Get additional data for each user
    const usersWithData = await Promise.all(
      users.map(async (user) => {
        // Get credits
      const creditsQuery = `SELECT 
    uid,
    btc,
    eth,
    ltc,
    bch,
    usdt,
    trx,
    doge,
    ada,
    xrp,
    bnb,
    usdp,
    nexo,
    mkr,
    tusd,
    usdc,
    busd,
    nc,
    inr,
    shib,
    matic,
    sc,
    mvr,
    bjb,
    aed,
    npr,
    pkr
FROM credits WHERE uid = $1`;
        const creditsResult = await pg.query(creditsQuery, [user.id]);
        const credits = creditsResult.rows[0] || {};

        // Get wager
        const wagerQuery = `SELECT wager FROM userwager WHERE uid = $1`;
        const wagerResult = await pg.query(wagerQuery, [user.id]);
        
        const wagerString = wagerResult.rows[0]?.wager || "0";
        const wager = parseFloat(wagerString.replace(/,/g, ""));

        // Calculate VIP level
        const vipDetails = getVipLevelDetails(wager);

        return {
          id: user.id,
          name: user.name,
          referalcode: user.referalcode,
          games_played: user.games_played,
          level: user.level,
          balance: credits.balance || 0,
          depositbalance: credits.depositbalance || 0,
          totaldeposited: credits.totaldeposited || 0,
          totalwithdrawed: credits.totalwithdrawed || 0,
          wager: wager,
          vip_level: vipDetails.vipLevel || 'VIP 00',
          card_type: vipDetails.card || 'brownz'
        };
      })
    );

    // Generate CSV content with headers
    const headers = [
      'ID', 'Name', 'Referral Code', 'Games Played', 'Level',
      'Balance', 'Deposit Balance', 'Total Deposited', 'Total Withdrawn',
      'Wager', 'VIP Level', 'Card Type'
    ];

    // Create CSV content
    let csvContent = headers.join(',') + '\r\n';

    usersWithData.forEach(user => {
      const row = [
        user.id,
        `"${user.name?.replace(/"/g, '""') || ''}"`, // Escape quotes in name
        `"${user.referalcode || ''}"`,
        user.games_played || 0,
        user.level || 0,
        user.balance || 0,
        user.depositbalance || 0,
        user.totaldeposited || 0,
        user.totalwithdrawed || 0,
        user.wager || 0,
        `"${user.vip_level || ''}"`,
        `"${user.card_type || ''}"`
      ];
      csvContent += row.join(',') + '\r\n';
    });

    // Set headers for CSV download
    const date = new Date().toISOString().split('T')[0];
    res.setHeader('Content-Type', 'text/csv');
    res.setHeader('Content-Disposition', `attachment; filename=user_reports_${date}.csv`);
    
    // Send CSV response
    return res.status(200).send(csvContent);
  } catch (error) {
    console.error('Error exporting users reports:', error);
    return res.status(500).json({ error: 'Failed to export users reports' });
  }
};


module.exports = {
    getUserReports,
    getAllUsersReports,
    exportUsersReports
  };