const axios = require('axios');
const { filterEnabledGames } = require('./gamefilter');
const moment = require('moment');


const APP_TZ = process.env.APP_TZ || 'Asia/Kolkata';

const SPORT_NAME_BY_ID = {
  "4": "Cricket",
  "2": "Tennis",
  "1": "Soccer",
  "6423": "American Football",
  "7511": "Baseball",
  "7522": "Basketball",
  "6": "Boxing",
  "3503": "Darts",
  "2152880": "Gaelic Games",
  "3": "Golf",
  "4339": "Greyhound Racing",
  "15": "Greyhound Todays Card",
  "13": "Horse Race Todays Card",
  "7": "Horse Racing",
  "26420387": "Mixed Martial Arts",
  "8": "Motor Sport",
  "1477": "Rugby League",
  "5": "Rugby Union",
  "6422": "Snooker"
};

const IST_OFFSET = 330;

function getRange(dateType) {
  console.log("🔍 getRange() called with:", dateType);

  // Get base date in IST
  const base = moment().utcOffset(IST_OFFSET).startOf('day');
  console.log("📅 Base IST date:", base.format());

  if (dateType === 'today') {
    return {
      start: base.clone(),
      end: base.clone().endOf('day')
    };
  }

  if (dateType === 'tomorrow') {
    const t = base.clone().add(1, 'day');
    return {
      start: t.clone(),
      end: t.clone().endOf('day')
    };
  }

  return null;
}

// --- Helper: earliest market start
function getEarliestMarketStart(game) {
  if (!game?.markets?.length) return null;
  let min = null;
  for (const m of game.markets) {
    const dt = m?.marketStartTime ? moment(m.marketStartTime) : null;
    if (dt?.isValid()) {
      if (!min || dt.isBefore(min)) min = dt;
    }
  }
  return min;
}
const getAllMatches = async (req, res) => {
  try {
    const response = await axios.get('http://89.116.20.218:8085/api/home', {
      headers: {
        'x-rapidapi-key': process.env.RAPIDAPI_KEY ?? 'eNp57Jb7yHgI26S8Yw4u_v31',
        'x-rapidapi-host': process.env.RAPIDAPI_HOST ?? 'betfair14.p.rapidapi.com',
        'X-ScoreSwift-Key': process.env.SCORESWIFT_KEY ?? 'eNp57Jb7yHgI26S8Yw4u_v31'
      }
    });

    // Filter only enabled games
    const filteredGames = await filterEnabledGames(response.data);

    res.json(filteredGames);
  } catch (error) {
    console.error('All matches fetch error:', error);
    res.status(500).json({ 
      error: 'Error fetching matches' 
    });
  }
};

const getMatchesByGameId = async (req, res) => {
  try {
    const { gameId } = req.params;

    const response = await axios.get('http://89.116.20.218:8085/api/home', {
      headers: {
        'x-rapidapi-key': process.env.RAPIDAPI_KEY ?? 'eNp57Jb7yHgI26S8Yw4u_v31',
        'x-rapidapi-host': process.env.RAPIDAPI_HOST ?? 'betfair14.p.rapidapi.com',
        'X-ScoreSwift-Key': process.env.SCORESWIFT_KEY ?? 'eNp57Jb7yHgI26S8Yw4u_v31'
      }
    });

    // Filter only enabled games
    const filteredGames = await filterEnabledGames(response.data);

    // Find matches for specific game ID
    const matchesForGame = filteredGames.find(game => game.id === gameId);

    if (!matchesForGame) {
      return res.status(404).json({ 
        error: 'No matches found for the specified game ID' 
      });
    }

    res.json(matchesForGame);
  } catch (error) {
    console.error('Matches by game ID fetch error:', error);
    res.status(500).json({ 
      error: 'Error fetching matches by game ID' 
    });
  }
};

const getMatchesByDate = async (req, res) => {
  try {
    const { dateType } = req.params; // 'today' or 'tomorrow'
    
    const response = await axios.get('http://89.116.20.218:8085/api/home', {
      headers: {
        'x-rapidapi-key': process.env.RAPIDAPI_KEY ?? 'eNp57Jb7yHgI26S8Yw4u_v31',
        'x-rapidapi-host': process.env.RAPIDAPI_HOST ?? 'betfair14.p.rapidapi.com',
        'X-ScoreSwift-Key': process.env.SCORESWIFT_KEY ?? 'eNp57Jb7yHgI26S8Yw4u_v31'
      }
    });

    // Filter only enabled games
    const filteredGames = await filterEnabledGames(response.data);

    // Determine date range
    const today = moment().startOf('day');
    const tomorrow = moment().add(1, 'days').startOf('day');

    const filteredMatchesByDate = filteredGames.filter(game => {
      const matchDate = moment(game.markets[0].marketStartTime);
      
      if (dateType === 'today') {
        return matchDate.isSame(today, 'day');
      } else if (dateType === 'tomorrow') {
        return matchDate.isSame(tomorrow, 'day');
      }
      
      return false;
    });

    // Additional flag for inplay matches
    const processedMatches = filteredMatchesByDate.map(game => ({
      ...game,
      isInPlay: game.markets.some(market => market.inplay === true)
    }));

    res.json(processedMatches);
  } catch (error) {
    console.error('Matches by date fetch error:', error);
    res.status(500).json({ 
      error: 'Error fetching matches by date' 
    });
  }
};


// const getMatchesByDateGame = async (req, res) =>{
//   const { dateType, gameId } = req.params;

//   if (!['today', 'tomorrow'].includes(dateType)) {
//     return res.status(400).json({ error: "dateType must be 'today' or 'tomorrow'" });
//   }

//   if (!/^\d+$/.test(gameId)) {
//     return res.status(400).json({ error: 'gameId must be a numeric string' });
//   }

//   const range = getRange(dateType);
//   if (!range) return res.status(400).json({ error: 'Invalid dateType' });

//   try {
//     // 1) fetch all games
//     const response = await axios.get('http://89.116.20.218:8085/api/home', {
//       headers: {
//         'x-rapidapi-key': process.env.RAPIDAPI_KEY ?? 'eNp57Jb7yHgI26S8Yw4u_v31',
//         'x-rapidapi-host': process.env.RAPIDAPI_HOST ?? 'betfair14.p.rapidapi.com',
//         'X-ScoreSwift-Key': process.env.SCORESWIFT_KEY ?? 'eNp57Jb7yHgI26S8Yw4u_v31'
//       },
//       timeout: 15000
//     });

//     const allGames = Array.isArray(response.data) ? response.data : [];
//     if (!allGames.length) return res.json([]);

//     // 2) filter enabled
//     const enabledGames = await filterEnabledGames(allGames);

//     // 3) filter by gameId
//     const sportGames = enabledGames.filter(g => String(g.id) === String(gameId));
//     if (!sportGames.length) return res.json([]);

//     // 4) filter by date
//     const { start, end } = range;
//     const byDate = sportGames.filter(g => {
//       const first = getEarliestMarketStart(g);
//       if (!first) return false;
//       const local = moment(first).tz(APP_TZ);
//       return local.isSameOrAfter(start) && local.isSameOrBefore(end);
//     });

//     // 5) decorate
//     const processed = byDate.map(g => ({
//       ...g,
//       sportId: String(gameId),
//       sportName: SPORT_NAME_BY_ID[String(gameId)] ?? g.name ?? 'Unknown',
//       isInPlay: Array.isArray(g.markets) && g.markets.some(m => m?.inplay === true)
//     }));

//     return res.json(processed);
//   } catch (error) {
//     console.error('Matches by game & date error:', error?.response?.data || error.message);
//     return res.status(500).json({ error: 'Error fetching matches by game and date' });
//   }
// }
const getMatchesByDateGame = async (req, res) => {
  console.log("✅ API Hit - Params Received:", req.params);

  const { dateType, gameId } = req.params;

  // Step 1: Validate dateType
  console.log("🛠 Validating dateType...");
  if (!['today', 'tomorrow'].includes(dateType)) {
    console.log("❌ Invalid dateType:", dateType);
    return res.status(400).json({ error: "dateType must be 'today' or 'tomorrow'" });
  }

  // Step 2: Validate gameId
  console.log("🛠 Validating gameId...");
  if (!/^\d+$/.test(gameId)) {
    console.log("❌ Invalid gameId:", gameId);
    return res.status(400).json({ error: 'gameId must be a numeric string' });
  }

  // Step 3: Get date range
  console.log("🛠 Calculating date range...");
  const range = getRange(dateType);
  console.log("📅 Date Range:", range);
  if (!range) return res.status(400).json({ error: 'Invalid dateType' });

  try {
    // Step 4: Fetch all games from API
    console.log("🌍 Fetching all games from external API...");
    const response = await axios.get('http://89.116.20.218:8085/api/home', {
      headers: {
        'x-rapidapi-key': process.env.RAPIDAPI_KEY ?? 'eNp57Jb7yHgI26S8Yw4u_v31',
        'x-rapidapi-host': process.env.RAPIDAPI_HOST ?? 'betfair14.p.rapidapi.com',
        'X-ScoreSwift-Key': process.env.SCORESWIFT_KEY ?? 'eNp57Jb7yHgI26S8Yw4u_v31'
      },
      timeout: 15000
    });

    console.log("📦 Raw API Data Length:", Array.isArray(response.data) ? response.data.length : "Not an array");

    const allGames = Array.isArray(response.data) ? response.data : [];
    if (!allGames.length) {
      console.log("⚠ No games returned from API");
      return res.json([]);
    }

    // Step 5: Filter enabled games
    console.log("🛠 Filtering enabled games from DB...");
    const enabledGames = await filterEnabledGames(allGames);
    console.log(`✅ Enabled Games Count: ${enabledGames.length}`);

    // Step 6: Filter by gameId
    console.log(`🛠 Filtering by gameId: ${gameId}...`);
    const sportGames = enabledGames.filter(g => String(g.id) === String(gameId));
    console.log(`✅ Games after gameId filter: ${sportGames.length}`);
    if (!sportGames.length) {
      console.log(`⚠ No games found for gameId ${gameId} after filtering`);
      return res.json([]);
    }

    
   // Step 7: Filter by date
const { start, end } = range;
console.log(`🛠 Filtering by date: ${start.format()} to ${end.format()}...`);
const byDate = sportGames.filter(g => {
  const first = getEarliestMarketStart(g);
  console.log(`📅 Game ID ${g.id} earliest start:`, first ? first.format() : "No date");
  if (!first) return false;
  const local = moment(first).utcOffset(330); // ✅ Convert to IST without moment-timezone
  return local.isSameOrAfter(start) && local.isSameOrBefore(end);
});
console.log(`✅ Games after date filter: ${byDate.length}`);


    // Step 8: Decorate final output
    console.log("🛠 Decorating games with extra info...");
    const processed = byDate.map(g => ({
      ...g,
      sportId: String(gameId),
      sportName: SPORT_NAME_BY_ID[String(gameId)] ?? g.name ?? 'Unknown',
      isInPlay: Array.isArray(g.markets) && g.markets.some(m => m?.inplay === true)
    }));

    console.log(`🎯 Final Games Count: ${processed.length}`);
    return res.json(processed);

  } catch (error) {
    console.error('❌ Matches by game & date error:', error?.response?.data || error.message);
    return res.status(500).json({ error: 'Error fetching matches by game and date' });
  }
};

module.exports = { 
  getAllMatches,
  getMatchesByGameId,
  getMatchesByDate ,
  getMatchesByDateGame
};