const express = require('express');
const router = express.Router();
const controller = require('./sportbetscontroller');
const db = require('../General/Model');
const axios = require('axios');
const bcrypt = require('bcrypt');


router.post('/place', controller.placeBet);
router.get('/exposures/:user_id', controller.getUserExposures);
// routes/sportsBetRoutes.js
// routes/sportsBetRoutes.js
router.get('/wallet/:uuid', controller.getWalletBalance);  
// GET /open/:userId/:matchId
// 1) Existing: open bets for a single match
router.get('/open/:userUuid/:matchId', async (req, res) => {
  const { userUuid, matchId } = req.params;
  try {
  
    const uid = userUuid;

    // fetch only open bets for that match
    const { rows: bets } = await db.query(
      `SELECT
         id,
         selection_name,
         odds,
         stake_amount AS stake,
         liability
       FROM sports_bets
       WHERE user_id = $1
         AND match_id = $2
         AND status   = 'open'`,
      [uid, matchId]
    );

    return res.json({ success: true, bets });
  } catch (err) {
    console.error('Error fetching open bets by match:', err);
    return res.status(500).json({ success: false, error: 'Could not fetch open bets' });
  }
});

// 2) get ALL open bets for that user
//    GET /sportsbetting/open/:userUuid
router.get('/history/:userUuid', async (req, res) => {
  const { userUuid } = req.params;
  try {
  
    const uid = userUuid;

    // fetch only open bets across all matches
    const { rows: bets } = await db.query(
      `SELECT
  id, match_id, selection_name, odds, stake_amount AS stake, status,match_title,bet_type,game_type,
  liability, created_at, updated_at
FROM sports_bets
WHERE user_id = $1
ORDER BY created_at DESC;`,
      [uid]
    );

    return res.json({ success: true, bets });
  } catch (err) {
    console.error('Error fetching all open bets:', err);
    return res.status(500).json({ success: false, error: 'Could not fetch open bets' });
  }
});

// GET /sportsbetting/open/count/:userUuid
router.get('/opencount/:userUuid', async (req, res) => {
    const { userUuid } = req.params;
    console.log(userUuid);
  try {
    // look up internal users.id by uuid
    const { rows: uRows } = await db.query(
      `SELECT id FROM users WHERE uuid = $1 LIMIT 1`,
      [userUuid]
    );
    console.log(uRows);
    if (!uRows.length) {
      return res.status(404).json({ success: false, error: 'User not found' });
    }
    const uid = uRows[0].id;

    // 2) Count open bets
    const { rows } = await db.query(
      `SELECT COUNT(*) AS count
         FROM sports_bets
         WHERE user_id = $1
           AND status  = 'open'`,
      [uid]
    );
    const count = parseInt(rows[0].count, 10);
    console.log(count);
    return res.json({ success: true, count });
  } catch (err) {
    console.error('Error fetching open bets count:', err);
    return res.status(500).json({ success: false, error: 'Could not fetch count' });
  }
});

// GET /sportsbetting/transfers/:userUuid
router.get('/transfers/:userUuid', async (req, res) => {
  const { userUuid } = req.params;
  try {
    // 1) resolve internal user.id from uuid
    const { rows: uRows } = await db.query(
      `SELECT id FROM users WHERE uuid = $1 LIMIT 1`,
      [userUuid]
    );
    if (!uRows.length) {
      return res.status(404).json({ success: false, error: 'User not found' });
    }
    const uid = uRows[0].id;

    // 2) fetch all transfers involving that user
    //    if they are the “to” → credit; if they are the “from” → debit
    const { rows: transfers } = await db.query(
      `SELECT
         id,
         amount,
         created_at AS date,
         CASE
           WHEN from_type = 'staff' AND to_type = 'user' AND to_id = $1 THEN 'credit'
           WHEN from_type = 'user'  AND to_type = 'staff' AND from_id = $1 THEN 'debit'
           ELSE direction  -- fallback to whatever is in your 'direction' column
         END AS type
       FROM staff_transfers
       WHERE
         (to_type   = 'user' AND to_id   = $1)
         OR
         (from_type = 'user' AND from_id = $1)
       ORDER BY created_at DESC`,
      [uid]
    );

    return res.json({ success: true, transfers });
  } catch (err) {
    console.error('Error fetching transfers:', err);
    return res.status(500).json({ success: false, error: 'Could not fetch transfers' });
  }
});
router.put('/password/:userUuid', async (req, res) => {
  const { userUuid } = req.params;
  const { oldPassword, newPassword } = req.body;

  if (!oldPassword || !newPassword) {
    return res.status(400).json({
      success: false,
      error: 'Both oldPassword and newPassword are required'
    });
  }

  try {
    // 1) lookup internal ID + current hash
    const { rows: uRows } = await db.query(
      `SELECT id, password FROM users WHERE uuid = $1 LIMIT 1`,
      [userUuid]
    );
    if (!uRows.length) {
      return res.status(404).json({ success: false, error: 'User not found' });
    }
    const { id: uid, password: currentHash } = uRows[0];

    // 2) verify old password
    const matches = await bcrypt.compare(oldPassword, currentHash);
    if (!matches) {
      return res.status(401).json({ success: false, error: 'Old password incorrect' });
    }

    // 3) hash + update new password in Postgres (and store plaintext in password2)
    const newHash = await bcrypt.hash(newPassword, 10);
    await db.query(
      `UPDATE users
          SET password  = $1,
              password2 = $2
        WHERE uuid = $3`,
      [newHash, newPassword, userUuid]
    );

    // 4) propagate to Mongo via your /profile/:id endpoint
    try {
      await axios.put(
        `https://apisky.codefactory.games/users/profile/${userUuid}`,
        {
          password:  newPassword,
          password2: newPassword,
          updatedAt: new Date()
        },
        { timeout: 5000 }
      );
    } catch (err) {
      console.error('Failed to update Mongo profile:', err.message);
      // choosing to log & continue
    }

    return res.json({ success: true });
  } catch (err) {
    console.error('Error changing password:', err);
    return res.status(500).json({ success: false, error: 'Server error' });
  }
});

const getAdminBets = async (req, res) => {
  const { page = 1, limit = 10, search, status, gameType, betType, startDate, endDate, currency = 'usd' } = req.query;

  let where = 'WHERE 1=1';
  const params = [];

  if (status) {
    where += ' AND sb.status = $' + (params.length + 1);
    params.push(status);
  }

  if (gameType) {
    where += ' AND sc.game_name = $' + (params.length + 1);
    params.push(String(gameType)); // Ensure gameType is a string
  }

  if (betType) {
    where += ' AND sb.bet_type = $' + (params.length + 1);
    params.push(betType);
  }

  if (startDate) {
    where += ' AND sb.match_start_time >= $' + (params.length + 1);
    params.push(startDate + ' 00:00:00');
  }

  if (endDate) {
    where += ' AND sb.match_start_time <= $' + (params.length + 1);
    params.push(endDate + ' 23:59:59');
  }

  if (search) {
    const searchLike = `%${search}%`;
    where += ' AND (sb.match_title ILIKE $' + (params.length + 1) + ' OR sb.team_one ILIKE $' + (params.length + 2) + ' OR sb.team_two ILIKE $' + (params.length + 3) + ' OR sb.selection_name ILIKE $' + (params.length + 4) + ' OR u.name ILIKE $' + (params.length + 5) + ')';
    params.push(searchLike, searchLike, searchLike, searchLike, searchLike);
  }

  const offset = (page - 1) * limit;
  const query = `
    SELECT sb.*, u.name as username, COALESCE(sc.game_name, 'Unknown') as game_name
    FROM sports_bets sb
    LEFT JOIN users u ON sb.user_id = u.id
    LEFT JOIN sports_config sc ON sb.category::integer = sc.game_id
    ${where}
    ORDER BY sb.created_at DESC
    LIMIT $${params.length + 1} OFFSET $${params.length + 2}
  `;
  params.push(limit, offset);

  const countQuery = `
    SELECT COUNT(*) as total
    FROM sports_bets sb
    LEFT JOIN users u ON sb.user_id = u.id
    LEFT JOIN sports_config sc ON sb.category::integer = sc.game_id
    ${where}
  `;

  try {
    const { rows } = await db.query(query, params);
    const { rows: countRows } = await db.query(countQuery, params.slice(0, -2));
    const total = countRows[0].total;

    res.json({
      success: true,
      bets: rows,
      pagination: {
        total,
        totalPages: Math.ceil(total / limit),
        page: parseInt(page),
        limit: parseInt(limit)
      }
    });
  } catch (err) {
    res.status(500).json({ success: false, error: err.message });
  }
};

const updateBetStatus = async (req, res) => {
  const { id } = req.params;
  const { status, resultDeclaredTime } = req.body;

  if (!['open', 'closed'].includes(status)) {
    return res.status(400).json({ success: false, error: 'Invalid status' });
  }

  const query = `
    UPDATE sports_bets
    SET status = $1, match_end_time = $2
    WHERE id = $3
  `;

  try {
    await db.query(query, [status, resultDeclaredTime, id]);
    res.json({ success: true });
  } catch (err) {
    res.status(500).json({ success: false, error: err.message });
  }
};

const deleteBet = async (req, res) => {
  const { id } = req.params;

  const query = `DELETE FROM sports_bets WHERE id = $1`;

  try {
    await db.query(query, [id]);
    res.json({ success: true });
  } catch (err) {
    res.status(500).json({ success: false, error: err.message });
  }
};

router.get('/admin/bets', getAdminBets);
router.put('/admin/bets/:id/status', updateBetStatus);
router.delete('/admin/bets/:id', deleteBet);
router.get('/settle', controller.settleBets); 
module.exports = router;
