const lib = require('xss-clean');
const db = require('../General/Model');
const fetch = require('node-fetch');

let cachedUsdRate = 0;
let cachedAt = 0;

async function inrToUsd(inrAmount = 0) {
  const FIVE_MIN = 5 * 60 * 1000;
  const now = Date.now();
  if (!cachedUsdRate || now - cachedAt > FIVE_MIN) {
    try {
      const r = await fetch('https://api.exchangerate.host/latest?base=INR&symbols=USD');
      const j = await r.json();
      cachedUsdRate = 1 / j.rates.USD;
      cachedAt = now;
    } catch (e) {
      console.error('FX fetch failed, using fallback 83:', e.message);
      cachedUsdRate = 1 / 83;
    }
  }
  return +(inrAmount * cachedUsdRate).toFixed(2);
}

const liabilityOf = (gameType, betType, stake, odds) => {
  if (gameType === 'FAN') return stake * odds / 100;
  return betType === 'lay' ? stake * (odds - 1) : stake;
};
exports.placeBet = async (req, res) => {
  const {
    user_id, game_type, match_id, match_title, selection_name,
    bet_type, odds, stake_amount, team_one, team_two,
    category, original_currency, original_amount, usd_amount, match_start_time
  } = req.body;
  console.log('PLACE BET:', {
    user_id, game_type, match_id, match_title, selection_name,
    bet_type, odds, stake_amount, team_one, team_two,
    category, original_currency, original_amount, usd_amount, match_start_time
  });

  const stake = Number(stake_amount);
  const oddN  = Number(odds);
  if (stake <= 0 || oddN <= (game_type === 'FAN' ? 0 : 1)) {
    return res.status(400).json({ success: false, error: 'Invalid stake or odds' });
  }

  // total liability for this single bet
  const liability = liabilityOf(game_type, bet_type, stake, oddN);
  const startTime = match_start_time || new Date().toISOString();

  try {
    await db.query('BEGIN');

    // 1) resolve user
    if(!user_id) {
      return res.status(400).json({ success: false, error: 'User ID is required' });
    }
   
    const uid = user_id;

    console.log('PLACE BET: resolved user_id:', uid);

    // 2) check wallet
    const { rows: cRows } = await db.query(
      `SELECT inr FROM credits WHERE uid = $1`,
      [uid]
    );
    if (!cRows[0]) throw new Error('Wallet not found');
    const currentInr = Number(cRows[0].inr);
    console.log('Current wallet balance:', currentInr);

    // 3) lock & fetch old exposures for this match
    const { rows: oldExpRows } = await db.query(`
      SELECT team_name, exposure_amount
        FROM user_exposures
       WHERE user_id = $1 AND match_id = $2
       FOR UPDATE
    `, [uid, match_id]);

    const oldExposures = {};
    oldExpRows.forEach(r => {
      oldExposures[r.team_name] = Number(r.exposure_amount);
    });

    console.log('OLD EXPOSURES:', oldExposures);

    // 4) compute new exposures
    const newExposures = { ...oldExposures };
    const otherTeam = selection_name === team_one ? team_two : team_one;

    if (game_type === 'MO') {
      // Match Odds: three‐way market
      if (bet_type === 'back') {
        const profit = stake * (oddN - 1);
        newExposures[selection_name] = (oldExposures[selection_name] || 0) + profit;
        newExposures[otherTeam]      = (oldExposures[otherTeam]        || 0) - stake;
        newExposures['The Draw']     = (oldExposures['The Draw']       || 0) - stake;
      } else { // lay
        const layLiab = stake * (oddN - 1);
        newExposures[selection_name] = (oldExposures[selection_name] || 0) - layLiab;
        newExposures[otherTeam]      = (oldExposures[otherTeam]      || 0) + stake;
        newExposures['The Draw']     = (oldExposures['The Draw']     || 0) + stake;
      }

    } else if (game_type === 'BM') {
      // Bookmaker: two‐way back/lay
      const delta = stake * (oddN - 1);
      newExposures[selection_name] =
        (oldExposures[selection_name] || 0) + (bet_type === 'back' ? delta : -delta);

    } else if (game_type === 'FAN') {
      // Fancy: yes/no, liability = stake * odds/100
      const fanLiab = stake * oddN / 100;
      newExposures[selection_name] =
        (oldExposures[selection_name] || 0) + (bet_type === 'back' ?  fanLiab : -fanLiab);

    } else {
      throw new Error(`Unknown game_type: ${game_type}`);
    }

    console.log('NEW EXPOSURES:', newExposures);

    // 5) Calculate liability change - Fixed Logic
    // Calculate total liability (negative exposures) before and after
    const oldExposureValues = Object.values(oldExposures);
    const newExposureValues = Object.values(newExposures);
    
    const oldLiabilities = oldExposureValues.filter(x => x < 0);
    const newLiabilities = newExposureValues.filter(x => x < 0);
    
    // Calculate max liability (most negative value becomes the liability amount)
    const oldTotalLiability = oldLiabilities.length > 0 ? Math.abs(Math.min(...oldLiabilities)) : 0;
    const newTotalLiability = newLiabilities.length > 0 ? Math.abs(Math.min(...newLiabilities)) : 0;
    
    console.log('LIABILITY CALCULATION:');
    console.log('- Old liabilities:', oldLiabilities);
    console.log('- New liabilities:', newLiabilities);
    console.log('- Old total liability (amount blocked):', oldTotalLiability);
    console.log('- New total liability (amount to block):', newTotalLiability);
    
    // Calculate the change in liability
    const liabilityIncrease = newTotalLiability - oldTotalLiability;
    
    console.log('- Liability change:', liabilityIncrease);
    
    let balanceChange = 0; // positive = deduct from balance, negative = add to balance
    
    if (Object.keys(oldExposures).length === 0) {
      // No old exposures - this is the first bet for this match
      balanceChange = newTotalLiability;
      console.log('- First bet for match, deducting:', balanceChange);
    } else {
      // There are old exposures
      if (liabilityIncrease > 0) {
        // Liability increased - deduct the increase from balance
        balanceChange = liabilityIncrease;
        console.log('- Liability increased, deducting:', balanceChange);
      } else if (liabilityIncrease < 0) {
        // Liability decreased - add the decrease back to balance
        balanceChange = liabilityIncrease; // This will be negative
        console.log('- Liability decreased, adding back:', Math.abs(liabilityIncrease));
      } else {
        // Liability stayed the same
        balanceChange = 0;
        console.log('- Liability unchanged, no balance change');
      }
    }
    
    // Check if user has sufficient balance for deduction
    if (balanceChange > 0 && currentInr < balanceChange) {
      throw new Error('Insufficient balance');
    }
    
    console.log('FINAL BALANCE CHANGE:', balanceChange, '(positive = deduct, negative = add back)');

    // 6) upsert exposures
    for (const [teamName, expoAmt] of Object.entries(newExposures)) {
      await db.query(`
        INSERT INTO user_exposures
          (user_id, match_id, team_name, exposure_amount, match_title)
        VALUES ($1,$2,$3,$4,$5)
        ON CONFLICT (user_id,match_id,team_name)
        DO UPDATE SET exposure_amount = EXCLUDED.exposure_amount,
                      updated_at = NOW()
      `, [uid, match_id, teamName, expoAmt, match_title]);
    }

    // 7) record the bet
    await db.query(`
      INSERT INTO sports_bets
       (user_id,game_type,match_id,match_title,team_one,team_two,
        selection_name,category,bet_type,odds,stake_amount,
        original_currency,original_amount,usd_amount,liability,
        match_start_time,exposure_after_bet,status)
      VALUES($1,$2,$3,$4,$5,$6,$7,$8,$9,$10,$11,$12,$13,$14,$15,$16,$17,$18)
    `, [
      uid, game_type, match_id, match_title, team_one, team_two,
      selection_name, category||1, bet_type, oddN, stake,
      original_currency||'INR', original_amount||stake, usd_amount||await inrToUsd(stake),
      liability, startTime, newTotalLiability, 'open'
    ]);

    // 8) adjust wallet
    if (balanceChange !== 0) {
      await db.query(
        `UPDATE credits SET inr = inr - $1 WHERE uid = $2`,
        [balanceChange, uid]
      );
      console.log('Wallet updated. Balance change:', balanceChange);
    } else {
      console.log('No wallet update needed');
    }

    await db.query('COMMIT');

    // Get updated balance for response
    const { rows: updatedRows } = await db.query(
      `SELECT inr FROM credits WHERE uid = $1`,
      [uid]
    );
    const newBalance = Number(updatedRows[0].inr);
    
    console.log('Transaction completed successfully');
    console.log('Old balance:', currentInr);
    console.log('New balance:', newBalance);
    console.log('Actual balance delta:', newBalance - currentInr);

    // 10) return exactly this shape
    return res.json({
      success: true,
      exposure: {
        [match_id]: {
          match_title,
          teams: newExposures
        }
      },
      balanceDelta: newBalance - currentInr
    });

  } catch (err) {
    await db.query('ROLLBACK');
    console.error('PLACE BET ERROR:', err.message);
    return res.status(400).json({ success:false, error:err.message });
  }
};
// exports.placeBet = async (req, res) => {
//   const {
//     user_id, game_type, match_id, match_title, selection_name,
//     bet_type, odds, stake_amount, team_one, team_two,
//     category, original_currency, original_amount, usd_amount, match_start_time
//   } = req.body;
//   console.log('PLACE BET:', {
//     user_id, game_type, match_id, match_title, selection_name,
//     bet_type, odds, stake_amount, team_one, team_two,
//     category, original_currency, original_amount, usd_amount, match_start_time
//   });

//   const stake = Number(stake_amount);
//   const oddN  = Number(odds);
//   if (stake <= 0 || oddN <= (game_type === 'FAN' ? 0 : 1)) {
//     return res.status(400).json({ success: false, error: 'Invalid stake or odds' });
//   }

//   // total liability for this single bet
//   const liability = liabilityOf(game_type, bet_type, stake, oddN);
//   const startTime = match_start_time || new Date().toISOString();

//   try {
//     await db.query('BEGIN');

//     // 1) resolve user
//     if(!user_id) {
//       return res.status(400).json({ success: false, error: 'User ID is required' });
//     }
   
//     const uid = user_id;

//     console.log('PLACE BET: resolved user_id:', uid);

//     // 2) check wallet
//     const { rows: cRows } = await db.query(
//       `SELECT inr FROM credits WHERE uid = $1`,
//       [uid]
//     );
//     if (!cRows[0]) throw new Error('Wallet not found');
//     const currentInr = Number(cRows[0].inr);
//     if (currentInr < liability) throw new Error('Insufficient balance');

//     // 3) lock & fetch old exposures for this match
//     const { rows: oldExpRows } = await db.query(`
//       SELECT team_name, exposure_amount
//         FROM user_exposures
//        WHERE user_id = $1 AND match_id = $2
//        FOR UPDATE
//     `, [uid, match_id]);

//     const oldExposures = {};
//     oldExpRows.forEach(r => {
//       oldExposures[r.team_name] = Number(r.exposure_amount);
//     });

//     // 4) compute new exposures
//     const newExposures = { ...oldExposures };
//     const otherTeam = selection_name === team_one ? team_two : team_one;

//     if (game_type === 'MO') {
//       // Match Odds: three‐way market
//       if (bet_type === 'back') {
//         const profit = stake * (oddN - 1);
//         newExposures[selection_name] = (oldExposures[selection_name] || 0) + profit;
//         newExposures[otherTeam]      = (oldExposures[otherTeam]        || 0) - stake;
//         newExposures['The Draw']     = (oldExposures['The Draw']       || 0) - stake;
//       } else { // lay
//         const layLiab = stake * (oddN - 1);
//         newExposures[selection_name] = (oldExposures[selection_name] || 0) - layLiab;
//         newExposures[otherTeam]      = (oldExposures[otherTeam]      || 0) + stake;
//         newExposures['The Draw']     = (oldExposures['The Draw']     || 0) + stake;
//       }

//     } else if (game_type === 'BM') {
//       // Bookmaker: two‐way back/lay
//       const delta = stake * (oddN - 1);
//       newExposures[selection_name] =
//         (oldExposures[selection_name] || 0) + (bet_type === 'back' ? delta : -delta);

//     } else if (game_type === 'FAN') {
//       // Fancy: yes/no, liability = stake * odds/100
//       const fanLiab = stake * oddN / 100;
//       newExposures[selection_name] =
//         (oldExposures[selection_name] || 0) + (bet_type === 'back' ?  fanLiab : -fanLiab);

//     } else {
//       throw new Error(`Unknown game_type: ${game_type}`);
//     }

//     // 5) compute effective debit (refund reduction in exposure)
//     console.log('5 ---------------OLD EXPOSURES:', oldExposures);
//     console.log('5 ---------------NEW EXPOSURES:', newExposures);
//     console.log('5 ---------------OLD EXPOSURES:', liability);
//     const oldMax = Math.max(0, ...Object.values(oldExposures).map(x=>Math.abs(x)));
//     const newMax = Math.max(0, ...Object.values(newExposures).map(x=>Math.abs(x)));
//     const refund   = Math.max(0, oldMax - newMax);
//     const debit    = liability - refund;

//     console.log('5 ---------------BET:', {oldMax, newMax, refund, debit});

//     // 6) upsert exposures
//     for (const [teamName, expoAmt] of Object.entries(newExposures)) {
//       await db.query(`
//         INSERT INTO user_exposures
//           (user_id, match_id, team_name, exposure_amount, match_title)
//         VALUES ($1,$2,$3,$4,$5)
//         ON CONFLICT (user_id,match_id,team_name)
//         DO UPDATE SET exposure_amount = EXCLUDED.exposure_amount,
//                       updated_at = NOW()
//       `, [uid, match_id, teamName, expoAmt, match_title]);
//     }

//     // 7) record the bet
//     await db.query(`
//       INSERT INTO sports_bets
//        (user_id,game_type,match_id,match_title,team_one,team_two,
//         selection_name,category,bet_type,odds,stake_amount,
//         original_currency,original_amount,usd_amount,liability,
//         match_start_time,exposure_after_bet,status)
//       VALUES($1,$2,$3,$4,$5,$6,$7,$8,$9,$10,$11,$12,$13,$14,$15,$16,$17,$18)
//     `, [
//       uid, game_type, match_id, match_title, team_one, team_two,
//       selection_name, category||1, bet_type, oddN, stake,
//       original_currency||'INR', original_amount||stake, usd_amount||await inrToUsd(stake),
//       liability, startTime, newMax, 'open'
//     ]);

//     // 8) adjust wallet
//     await db.query(
//       `UPDATE credits SET inr = inr - $1 WHERE uid = $2`,
//       [debit, uid]
//     );

//     await db.query('COMMIT');

//     // 10) return exactly this shape
//     return res.json({
//       success: true,
//       exposure: {
//         [match_id]: {
//           match_title,
//           teams: newExposures
//         }
//       },
//       balanceDelta: -debit
//     });

//   } catch (err) {
//     await db.query('ROLLBACK');
//     console.error('PLACE BET ERROR:', err.message);
//     return res.status(400).json({ success:false, error:err.message });
//   }
// };

// GET /sportsbetting/exposures/:user_id
exports.getUserExposures = async (req,res) => {
  const { user_id } = req.params;
  try {
    const { rows:u } = await db.query(
      `SELECT id FROM users WHERE id=$1`, [user_id]
    );
    if (!u.length) return res.status(404).json({ success:false, error:'User not found' });
    const uid = u[0].id;

    const { rows } = await db.query(`
      SELECT ue.match_id, ue.team_name, ue.exposure_amount,
             (
               SELECT sb.match_title
                 FROM sports_bets sb
                WHERE sb.match_id=ue.match_id
                LIMIT 1
             ) AS match_title
        FROM user_exposures ue
       WHERE ue.user_id=$1
    `, [uid]);

    const out = {};
    for (const r of rows) {
      out[r.match_id] = out[r.match_id]||{match_title:r.match_title,teams:{}};
      out[r.match_id].teams[r.team_name] = +r.exposure_amount;
    }
    return res.json({ success:true, exposures: out });
  } catch (err) {
    console.error(err);
    return res.status(500).json({ success:false, error:'Could not fetch exposures' });
  }
};

// GET /sportsbetting/open/:user_id/:match_id
exports.getOpenBets = async (req,res) => {
  const { user_id, match_id } = req.params;
  try {
    const { rows:u } = await db.query(`SELECT id FROM users WHERE uuid=$1`, [user_id]);
    if (!u.length) return res.status(404).json({ success:false, error:'User not found' });
    const uid = u[0].id;

   const { rows: bets } = await db.query(
  `SELECT
     id,
     user_id,
     game_type,
     match_title,
     team_one,
     team_two,
     selection_name,
     category,
     bet_type,
     odds,
     stake_amount,
     original_currency,
     original_amount,
     usd_amount,
     liability,
     match_start_time,
     match_end_time,
     created_at,
     updated_at,
     match_id,
     exposure_after_bet,
     status
   FROM sports_bets
   WHERE user_id    = $1
     AND match_id   = $2
     AND status     = 'open'`,
  [uid, match_id]
);


    return res.json({ success:true, bets });
  } catch (err) {
    console.error(err);
    return res.status(500).json({ success:false, error:'Could not fetch open bets' });
  }
};

// exports.getUserExposures = async (req, res) => {
//   const { user_id } = req.params; // this is the UUID

//   try {
//     // 1) resolve UUID → internal bigint user id
//     const { rows: uRows } = await db.query(
//       `SELECT id
//          FROM users
//         WHERE uuid = $1
//         LIMIT 1`,
//       [user_id]
//     );
//     if (!uRows.length) {
//       return res.status(404).json({ success: false, error: 'User not found' });
//     }
//     const uid = uRows[0].id;

//     // 2) pull every team-level exposure for that user, with the match_title
//     const { rows } = await db.query(
//       `SELECT
//          ue.match_id,
//          ue.team_name,
//          ue.exposure_amount,
//          -- grab any one match_title from sports_bets for display
//          ( SELECT sb.match_title
//              FROM sports_bets sb
//             WHERE sb.match_id = ue.match_id
//             LIMIT 1
//          ) AS match_title
//        FROM user_exposures ue
//       WHERE ue.user_id = $1`,
//       [uid]
//     );

//     // 3) fold into { [match_id]: { match_title, teams: { team_name: exposure, … } } }
//     const exposuresByMatch = {};
//     for (const { match_id, team_name, exposure_amount, match_title } of rows) {
//       if (!exposuresByMatch[match_id]) {
//         exposuresByMatch[match_id] = {
//           match_title,
//           teams: {}
//         };
//       }
//       exposuresByMatch[match_id].teams[team_name] = +exposure_amount;
//     }

//     return res.json({
//       success: true,
//       exposures: exposuresByMatch
//     });

//   } catch (err) {
//     console.error('getUserExposures error:', err);
//     return res.status(500).json({ success: false, error: 'Could not fetch exposures' });
//   }
// };
exports.getWalletBalance = async (req, res) => {
  const { uuid } = req.params;
  try {
    const { rows: uRows } = await db.query(`SELECT id FROM users WHERE id = $1 LIMIT 1`, [uuid]);
    if (!uRows.length) return res.status(404).json({ success: false, error: 'User not found' });

    const uid = uRows[0].id;
    const { rows: cRows } = await db.query(`SELECT inr FROM credits WHERE uid = $1 LIMIT 1`, [uid]);
    if (!cRows.length) return res.status(404).json({ success: false, error: 'Wallet not found' });

    return res.json({ success: true, balance: +cRows[0].inr });
  } catch (e) {
    console.error('getWalletBalance error:', e);
    res.status(500).json({ success: false, error: 'Server error' });
  }
};

exports.settleBets = async () => {
  try {
    // Fetch matches with declared results that haven't been settled
    const { rows: matches } = await db.query(`
      SELECT id, result, team_one, team_two
      FROM matches
      WHERE result IS NOT NULL AND result_declared_at IS NOT NULL
    `);

    for (const match of matches) {
      const { id: match_id, result, team_one, team_two } = match;

      // Fetch all open bets for this match
      const { rows: bets } = await db.query(`
        SELECT id, user_id, game_type, bet_type, odds, stake_amount, selection_name, team_one, team_two, liability
        FROM sports_bets
        WHERE match_id = $1 AND status = 'open'
      `, [match_id]);

      for (const bet of bets) {
        await db.query('BEGIN');
        try {
          const { user_id, game_type, bet_type, odds, stake_amount, selection_name, liability } = bet;
          let payout = 0;

          if (game_type === 'MO') {
            if (bet_type === 'back' && selection_name === result) {
              payout = stake_amount * odds; // Win: stake * odds
            } else if (bet_type === 'lay' && selection_name !== result) {
              payout = stake_amount; // Win: return stake
            } else {
              payout = 0; // Loss
            }
          } else if (game_type === 'BM') {
            if (bet_type === 'back' && selection_name === result) {
              payout = stake_amount * (odds - 1); // Win: profit only
            } else if (bet_type === 'lay' && selection_name !== result) {
              payout = stake_amount; // Win: return stake
            } else {
              payout = 0; // Loss
            }
          } else if (game_type === 'FAN') {
            if (bet_type === 'back' && selection_name === result) {
              payout = stake_amount * odds / 100; // Win: stake * odds/100
            } else if (bet_type === 'lay' && selection_name !== result) {
              payout = stake_amount; // Win: return stake
            } else {
              payout = 0; // Loss
            }
          }

          // Update user balance
          if (payout > 0) {
            await db.query(`UPDATE credits SET inr = inr + $1 WHERE uid = $2`, [payout, user_id]);
            await fetch(`https://apisky.codefactory.games/users/balance/${user_id}`, {
              method: 'PUT',
              headers: { 'Content-Type': 'application/json' },
              body: JSON.stringify({ type: 'deposit', amount: payout })
            }).catch(err => console.error('Mongo sync failed:', err));
          }

          // Mark bet as settled
          await db.query(`UPDATE sports_bets SET status = 'settled', updated_at = NOW() WHERE id = $1`, [bet.id]);

          // Clear exposures for this user and match
          await db.query(`
            DELETE FROM user_exposures
            WHERE user_id = $1 AND match_id = $2
          `, [user_id, match_id]);

          await db.query('COMMIT');
        } catch (err) {
          await db.query('ROLLBACK');
          console.error(`Failed to settle bet ${bet.id}:`, err.message);
        }
      }

      // Update match to mark as settled
      await db.query(`UPDATE matches SET updated_at = NOW() WHERE id = $1`, [match_id]);
    }
  } catch (err) {
    console.error('settleBets error:', err);
  }
};