// controllers/staffController.js
const pg       = require('../../General/Model');
const bcrypt   = require('bcrypt');
const config  = require("../../config");
const { transferInr } = require('../utils/transfer');
const {visibleUserIds} = require('../helpers/staffTree');
const _ = require("lodash");
const BALANCE = config.BALANCE;
const WALLET  = config.WALLET;
// ───── helpers ────────────────────────────────────────────────────────────────
const getLevel = async (roleId) =>
  (await pg.query('SELECT level FROM roles WHERE id=$1', [roleId])).rows[0].level;

const allowedChild = (parentLvl, targetLvl) =>
  parentLvl === 0           // Super-Admin can create anything
    ? true
    : targetLvl > parentLvl;   // any deeper level is fine

const isInfinite = (staff) => staff.level === 0;

/** verify that `target` sits somewhere below `ancestor` in closure table */
const assertDescendant = async (pg, ancestorId, targetId) => {
  const ok = (
    await pg.query(
      'SELECT 1 FROM staff_hierarchy WHERE ancestor_id=$1 AND descendant_id=$2',
      [ancestorId, targetId]
    )
  ).rows.length;
  if (!ok) throw new Error('Target is not in your hierarchy');
};
async function generateNumericId() {
  while (true) {
    const candidate = _.random(1_000_000_000, 9_999_999_999);
    const hit = await pg.query("SELECT 1 FROM users WHERE id=$1", [candidate]);
    if (!hit.rowCount) return candidate;           // free – use it
  }
}

/* Helper: generate unique referral code (5 random chars after 3-char prefix) */
async function generateReferralCode(name) {
  const alphabet = "ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz0123456789";

  while (true) {
    const suffix = _.sampleSize(alphabet, 5).join("");
    const code   = `${name.slice(0, 3)}${suffix}`;
    const dup    = await pg.query("SELECT 1 FROM users WHERE referalcode=$1", [code]);
    if (!dup.rowCount) return code;                // code is unique
  }
}
// ───── CRUD & business logic ─────────────────────────────────────────────────

// 1. CREATE STAFF (levels 1-5)
exports.createStaff = async (req, res, next) => {
  
  await pg.query('BEGIN');
  try {
    const creator = req.staff;            // {id, role_id, level}
    const {
      name, email, phone, country,
      password, role_name,
      percentage = 0, 
      initial_balance = 0,
      parent_id      // optional: explicit anchor
    } = req.body;
    console.log("createStaff", { creator, body: req.body });
    const { rows: [role] } = await pg.query(
      'SELECT id, level FROM roles WHERE name=$1', [role_name]
    );
    if (!role) return res.status(400).json({ status:'fail', message:'Bad role_name' });

    // one-step hierarchy rule
    if (!allowedChild(creator.level, role.level))
      return res.status(403).json({ status:'fail', message:'Hierarchy violation' });

    const anchor = parent_id || creator.id;
    if (creator.level !== 0)            // SuperAdmin may attach anywhere
      await assertDescendant(pg, creator.id, anchor);

    // balance check
    if (!isInfinite(creator) && initial_balance > 0) {
      const bal = (
        await pg.query(
          'SELECT inr FROM staff_balances WHERE staff_id=$1', [creator.id]
        )
      ).rows[0]?.inr || 0;
      if (initial_balance > bal)
        return res.status(400).json({ status:'fail', message:'Insufficient INR' });
    }
   console.log(role);
    const hash = await bcrypt.hash(password, 10);
    const { rows:[newStaff] } = await pg.query(
      `INSERT INTO staff
         (name,email,password,password2,phone,country,role_id,parent_id,percentage)
       VALUES ($1,$2,$3,$4,$5,$6,$7,$8,$9)
       RETURNING id`,
      [name,email,hash,password,phone,country,role.id,anchor,percentage]
    );
  /* ─── WhatsApp referral link ─────────────────────────────── */
   const slug = await require('../utils/genSlug')();    // e.g. 'ab3k92x'
  await pg.query(
    `INSERT INTO staff_whatsapp_ref (staff_id, slug, phone)
     VALUES ($1,$2,$3)`, [newStaff.id, slug, phone]
  );
    await pg.query(
      'INSERT INTO staff_balances (staff_id,inr) VALUES ($1,0)',
      [newStaff.id]
    );
 if (initial_balance > 0) {
    await transferInr(
      pg, 'staff', creator.id,
      'staff', newStaff.id,
      initial_balance,
      isInfinite(creator)
    );
  }
    // closure rows
    await pg.query(
      'INSERT INTO staff_hierarchy (ancestor_id,descendant_id,depth) VALUES ($1,$1,0)  ON CONFLICT DO NOTHING',
      [newStaff.id]
    );
    await pg.query(
      'INSERT INTO staff_hierarchy (ancestor_id,descendant_id,depth) VALUES ($1,$2,1)  ON CONFLICT DO NOTHING',
      [anchor, newStaff.id]
    );
    await pg.query(
      `INSERT INTO staff_hierarchy (ancestor_id,descendant_id,depth)
         SELECT ancestor_id,$1,depth+1
           FROM staff_hierarchy
          WHERE descendant_id=$2  ON CONFLICT DO NOTHING`,
      [newStaff.id, anchor]
    );

    await pg.query('COMMIT');
    res.status(201).json({ status:'success', staffId:newStaff.id, wa_link:`https://api.boss707.com/?ref=${slug}` });
  } catch (err) {
    await pg.query('ROLLBACK');
    next(err);
  } finally { }
};



// 2. CREATE PLAYER (level 6)
// exports.createPlayer = async (req, res, next) => {

//   await pg.query("BEGIN");

//   try {
//     const creator = req.staff;                     // injected by auth middleware
//     console.log("createPlayer", { creator, body: req.body });
//   if (creator.level === 6)
//   return res.status(403).json({ status:"fail", message:"Not allowed" });

//     const {
//       username, email, phone, country,
//       password, initial_balance = 0,
//       parent_id                    // optional explicit anchor
//     } = req.body;

//     /* collision checks ----------------------------------------------------- */
//     const dupeUser = await pg.query(
//       `SELECT 1 FROM users WHERE lower(email)=lower($1) OR lower(name)=lower($2)`,
//       [email, username]
//     );
//     if (dupeUser.rowCount)
//       return res.status(400).json({ status:"fail", message:"E-mail or username already exists" });

//     /* INR check (unless infinite) ----------------------------------------- */
//     if (!isInfinite(creator) && initial_balance > 0) {
//       const { rows:[{ inr=0 } = {}] } =
//         await pg.query("SELECT inr FROM staff_balances WHERE staff_id=$1", [creator.id]);
//       if (initial_balance > inr)
//         return res.status(400).json({ status:"fail", message:"Insufficient INR" });
//     }

//     /* generate UID + referral code + hash --------------------------------- */
//     const newId        = await generateNumericId();
//     const referalCode  = await generateReferralCode(username);
//     const referralLink = `https://api.boss707.com/referal/${referalCode}`;
//     const hash         = await bcrypt.hash(password, 10);
//     const anchor       = parent_id || creator.id;

//     /* insert user ---------------------------------------------------------- */
//     const { rows:[u] } = await pg.query(
//       `INSERT INTO users
//          (id, name, email, password, password2,
//           phone, country, parent_staff_id, role_id,
//           friends, wallet, profit_low, profit_high, profit, referalcode, referral_link)
//        VALUES
//          ($1,$2,$3,$4,$5,
//           $6,$7,$8, 6,
//           $9,$10,$11,$12,$13,$14,$15)
//        RETURNING id`,
//       [
//         newId, username, email, hash, password,
//         phone, country, anchor,
//         "Support,",                         // friends (string)
//         JSON.stringify(WALLET),
//         JSON.stringify(BALANCE),            // profit_low
//         JSON.stringify(BALANCE),            // profit_high
//         JSON.stringify(BALANCE),            // profit
//         referalCode, referralLink
//       ]
//     );

//     /* zero credit row ------------------------------------------------------ */
//     await pg.query("INSERT INTO credits (uid,inr) VALUES ($1,0)", [u.id]);

//     /* INR transfer (creator → new player) ---------------------------------- */
//      if (initial_balance > 0) {
//     await transferInr(
//       pg, "staff", creator.id,
//       "user",  u.id,
//       initial_balance,
//       isInfinite(creator)
//     );
//   }
//     await pg.query("COMMIT");
//     res.status(201).json({ status:"success", userId:u.id });

//   } catch (err) {
//     await pg.query("ROLLBACK");
//     next(err);
//   } finally {
   
//   }
// };
exports.createPlayer = async (req, res, next) => {
  await pg.query("BEGIN");
  try {
    const creator = req.staff; // injected by auth middleware
    console.log("createPlayer", { creator, body: req.body });
    
    if (creator.level === 6)
      return res.status(403).json({ status: "fail", message: "Not allowed" });

    const {
      username, email, phone, country,
      password, initial_balance = 0,
      parent_id // optional explicit anchor
    } = req.body;

    /* collision checks ----------------------------------------------------- */
    const dupeUser = await pg.query(
      `SELECT 1 FROM users WHERE lower(email)=lower($1) OR lower(name)=lower($2)`,
      [email, username]
    );
    if (dupeUser.rowCount)
      return res.status(400).json({ status: "fail", message: "E-mail or username already exists" });

    /* INR check (unless infinite) ----------------------------------------- */
    if (!isInfinite(creator) && initial_balance > 0) {
      const { rows: [{ inr = 0 } = {}] } =
        await pg.query("SELECT inr FROM staff_balances WHERE staff_id=$1", [creator.id]);
      if (initial_balance > inr)
        return res.status(400).json({ status: "fail", message: "Insufficient INR" });
    }

    /* generate UID + referral code + hash --------------------------------- */
    const newId = await generateNumericId();
    const referalCode = await generateReferralCode(username);
    const referralLink = `https://api.boss707.com/referal/${referalCode}`;
    const hash = await bcrypt.hash(password, 10);
    const anchor = parent_id || creator.id;

    /* insert user ---------------------------------------------------------- */
    const { rows: [u] } = await pg.query(
      `INSERT INTO users 
         (id, name, email, password, password2,
          phone, country, parent_staff_id, role_id,
          friends, wallet, profit_low, profit_high, profit, referalcode, referral_link)
       VALUES 
         ($1,$2,$3,$4,$5,
          $6,$7,$8, 6,
          $9,$10,$11,$12,$13,$14,$15)
       RETURNING id`,
      [
        newId, username, email, hash, password,
        phone, country, anchor,
        "Support,", // friends (string)
        JSON.stringify(WALLET),
        JSON.stringify(BALANCE), // profit_low
        JSON.stringify(BALANCE), // profit_high
        JSON.stringify(BALANCE), // profit
        referalCode, referralLink
      ]
    );

    console.log("User created successfully for user ID:", u.id);

    /* zero credit row ------------------------------------------------------ */
    await pg.query("INSERT INTO credits (uid,inr) VALUES ($1,0)", [u.id]);
    console.log("Inserted into credits table successfully for user ID:", u.id);

    /* insert house entry --------------------------------------------------- */
    await pg.query(
      "INSERT INTO house(uid, max) VALUES($1, $2)",
      [u.id, 2]
    );
    console.log("Inserted into house table successfully for user ID:", u.id);

    /* fetch register bonus from siteconfig -------------------------------- */
    const registerBonusResult = await pg.query(
      `SELECT COALESCE((SELECT registerbonus FROM siteconfig LIMIT 1), 0) AS registerbonus`
    );
    const registerBonus = registerBonusResult.rows[0].registerbonus;
    console.log("Register bonus fetched:", registerBonus);

    /* initialize userbonus entry ------------------------------------------- */
    await pg.query(
      `INSERT INTO userbonus (userid, name, totalbonus, vipbonus, specialbonus, generalbonus, joiningbonus, createdat, updatedat) 
       VALUES ($1, $2, $3, 0, $4, 0, $5, CURRENT_TIMESTAMP, CURRENT_TIMESTAMP)`,
      [u.id, username, registerBonus, registerBonus, registerBonus]
    );
    console.log("User bonus initialized for user ID:", u.id);

    /* update credits with register bonus ----------------------------------- */
    await pg.query(
      `UPDATE credits 
       SET bjb = bjb + $2 
       WHERE uid = $1`,
      [u.id, registerBonus]
    );
    console.log("Credits updated with register bonus for user ID:", u.id);

    /* insert into bonushistory ---------------------------------------------- */
    await pg.query(
      `INSERT INTO bonushistory (userid, event, amount, createdat, updatedat) 
       VALUES ($1, $2, $3, CURRENT_TIMESTAMP, CURRENT_TIMESTAMP)`,
      [u.id, 'registerbonus', registerBonus]
    );
    console.log("Bonus history inserted for user ID:", u.id);

    /* initialize bonusgame entry -------------------------------------------- */
    await pg.query(
      `INSERT INTO bonusgame (userid, luckyspin, dailybonus, weeklybonus, monthlybonus, depositbonus, rollcompetitionbonus, createdat, updatedat, rakebackbonus) 
       VALUES ($1, 0, 0, 0, 0, 0, 0, CURRENT_TIMESTAMP, CURRENT_TIMESTAMP, 0)`,
      [u.id]
    );
    console.log("Bonus game initialized for user ID:", u.id);

    /* add additional entries to bonushistory ------------------------------- */
    await pg.query(
      `INSERT INTO bonushistory (userid, event, amount, createdat, updatedat) 
       VALUES 
       ($1, 'userbonus', 0, CURRENT_TIMESTAMP, CURRENT_TIMESTAMP),
       ($1, 'bonusgame', 0, CURRENT_TIMESTAMP, CURRENT_TIMESTAMP)`,
      [u.id]
    );
    console.log("Added entries to bonus history for user ID:", u.id);

    /* INR transfer (creator → new player) ---------------------------------- */
    if (initial_balance > 0) {
      await transferInr(
        pg, "staff", creator.id,
        "user", u.id,
        initial_balance,
        isInfinite(creator)
      );
    }

    await pg.query("COMMIT");
    res.status(201).json({ status: "success", userId: u.id });

  } catch (err) {
    await pg.query("ROLLBACK");
    console.log("Error in createPlayer:", err);
    next(err);
  } finally {
    // cleanup if needed
  }
};
/* ---------------------------------------------------------------
 * PATCH /api/staff/players/:id
 * -------------------------------------------------------------- */
exports.updatePlayer = async (req, res, next) => {
  try {
    const playerId = Number(req.params.id);
    const staffId  = req.staff.id;               // set by auth-middleware

    /* 1 ─ permissions: the player must be in MY down-line */
    const allowed = await visibleUserIds(staffId);        // [uid …]
    console.log(allowed);
    if (!allowed.includes(String(playerId)))
      return res.status(403).json({ error: 'Forbidden' });

    /* 2 ─ collect the editable fields that were sent */
    const {
      username,
      email,
      phone,
      country,
      password,
      parent_id          // optional new parent staff
    } = req.body;

    const sets  = [];          // SQL SET fragments
    const vals  = [playerId];  // $1 = id
    let   idx   = 2;           // next placeholder number

    if (username) { sets.push(`name  = $${idx}`); vals.push(username); idx++; }
    if (email)    { sets.push(`email = $${idx}`); vals.push(email);    idx++; }
    if (phone)    { sets.push(`phone = $${idx}`); vals.push(phone);    idx++; }
    if (country)  { sets.push(`country = $${idx}`); vals.push(country);idx++; }

    if (parent_id) {
      sets.push(`parent_staff_id = $${idx}`);     // bigint column
      vals.push(Number(parent_id));               // make sure it’s numeric
      idx++;
    }

    if (password) {
      const hash = await bcrypt.hash(password, 10);
      sets.push(`password  = $${idx}`);            vals.push(hash);     idx++;
      sets.push(`password2 = $${idx}`);            vals.push(password); idx++;
    }

    if (!sets.length)
      return res.status(400).json({ error: 'Nothing to update' });

    /* 3 ─ run update */
    const sql = `
      UPDATE users
         SET ${sets.join(', ')}
       WHERE id = $1
       RETURNING id`;

    const { rows:[row] } = await pg.query(sql, vals);

    res.json({ success:true, userId: row.id });
  } catch (err) {
    next(err);
  }
};

// 3. LIST STAFF IN YOUR TREE
// 4. LIST STAFF  (staff + players, with parent_id on every row)
// exports.listStaff = async (req, res, next) => {
//   try {
//     /* ── 1. All descendants that are STAFF ─────────────────────────────── */
//     const { rows: staffRows } = await pg.query(
//       `SELECT s.id,
//               s.name,
//               s.email,
//               r.name              AS role,
//               s.phone,
//               s.country,
//               sb.inr              AS balance,
//               s.parent_id         AS parent_id          -- ★ add parent_id
//          FROM staff_hierarchy h
//          JOIN staff s           ON s.id = h.descendant_id
//          JOIN roles r           ON r.id = s.role_id
//          LEFT JOIN staff_balances sb ON sb.staff_id = s.id
//         WHERE h.ancestor_id = $1
//       ORDER BY s.id`,
//       [req.staff.id]
//     );

//     /* ── 2. Players that belong to one of those staff IDs ──────────────── */
//     const staffIds = staffRows.map(r => r.id);
//     const { rows: playerRows } = await pg.query(
//       `SELECT u.id,
//               u.name,
//               u.email,
//               'User'              AS role,
//               u.phone,
//               u.country,
//               c.inr               AS balance,
//               u.parent_staff_id   AS parent_id          -- ★ alias to parent_id
//          FROM users u
//     LEFT JOIN credits c ON c.uid = u.id
//         WHERE u.parent_staff_id = ANY($1::bigint[])
//       ORDER BY u.id`,
//       [staffIds]
//     );

//     res.json({
//       status: "success",
//       data  : [...staffRows, ...playerRows]
//     });
//   } catch (err) { next(err); }
// };
/* ===============================================================
   List every STAFF ↓-tree of the current operator **plus** players
   Each returned row now has the same JSON shape, including
     • wa_slug   – WhatsApp referral slug  (NULL for players)
     • wa_phone  – WhatsApp phone number   (NULL for players)
   =============================================================== */
exports.listStaff = async (req, res, next) => {
  try {
    /* ────────────────────────────────────────────────────────────
       1.  All STAFF that sit under the authenticated operator
       ──────────────────────────────────────────────────────────── */
    const { rows: staffRows } = await pg.query(
      `
      SELECT  s.id,
              s.name,
              s.email,
              r.name                 AS role,          -- "Admin", …
              s.phone,
              s.country,
              sb.inr                 AS balance,
              s.parent_id,
              s.percentage,
              /* WhatsApp referral data (may be NULL) */
              w.slug                 AS wa_slug,
              w.phone                AS wa_phone
      FROM    staff_hierarchy         h
      JOIN    staff                   s  ON s.id = h.descendant_id
      JOIN    roles                   r  ON r.id = s.role_id
      LEFT JOIN staff_balances        sb ON sb.staff_id = s.id
      LEFT JOIN staff_whatsapp_ref    w  ON w.staff_id = s.id
      WHERE   h.ancestor_id = $1
      ORDER BY s.id
      `,
      [req.staff.id]
    );

    /* ────────────────────────────────────────────────────────────
       2.  All PLAYERS directly owned by those staff members
           (Players never have a WhatsApp slug of their own)
       ──────────────────────────────────────────────────────────── */
    const staffIds = staffRows.map(r => r.id);

    const { rows: playerRows } = await pg.query(
      `
      SELECT  u.id,
              u.name,
              u.email,
              'User'                 AS role,
              u.phone,
              u.country,
              c.inr                  AS balance,
              u.parent_staff_id      AS parent_id,

              NULL::text            AS wa_slug,       -- keep JSON shape
              NULL::text            AS wa_phone
      FROM    users                  u
      LEFT JOIN credits              c ON c.uid = u.id
      WHERE   u.parent_staff_id = ANY($1::bigint[])
      ORDER BY u.id
      `,
      [staffIds]
    );

    /* ────────────────────────────────────────────────────────────
       3.  Respond
       ──────────────────────────────────────────────────────────── */
    res.json({
      status: 'success',
      data  : [...staffRows, ...playerRows]
    });

  } catch (err) {
    next(err);
  }
};

// exports.listStaff = async (req, res, next) => {
//   try {
//     /* 1. all descendant staff rows ---------------------------------------- */
//     const { rows: staffRows } = await pg.query(
//       `SELECT s.id, s.name, s.email,
//               r.name  AS role,
//               s.phone, s.country,
//               sb.inr   AS balance
//          FROM staff_hierarchy h
//          JOIN staff s           ON s.id = h.descendant_id
//          JOIN roles r           ON r.id = s.role_id
//          LEFT JOIN staff_balances sb ON sb.staff_id = s.id
//         WHERE h.ancestor_id = $1
//         ORDER BY s.id`,
//       [req.staff.id]
//     );

//     /* 2. collect all those staff IDs to fetch their players --------------- */
//     const staffIds = staffRows.map(r => r.id);
//     const { rows: playerRows } = await pg.query(
//       `SELECT u.id,
//               u.name,
//               u.email,
//               'User'      AS role,
//               u.phone,
//               u.country,
//               c.inr       AS balance
//          FROM users u
//          LEFT JOIN credits c ON c.uid = u.id
//         WHERE u.parent_staff_id = ANY($1::bigint[])
//         ORDER BY u.id`,
//       [staffIds]
//     );

//     /* 3. concatenate and send back --------------------------------------- */
//     res.json({
//       status: "success",
//       data  : [...staffRows, ...playerRows]      // same array shape as before
//     });

//   } catch (err) { next(err); }
// };
// 4. LIST PLAYERS IN YOUR TREE
exports.listPlayers = async (req, res, next) => {
  try {
    const players = await pg.query(
      `SELECT u.id, u.name, u.email, c.inr
         FROM users u
         JOIN credits c ON c.uid = u.id
         JOIN staff_hierarchy h ON h.descendant_id = u.parent_staff_id
        WHERE h.ancestor_id = $1
        ORDER BY u.id`,
      [req.staff.id]
    );
    res.json({ status:'success', data:players.rows });
  } catch (err) { next(err); }
};


// ─── GET TREE  (root + recursive descendants, properly ordered) ────────────
// exports.getTree = async (req, res, next) => {
//   try {
//     /*  The ‘path’ array lets us sort rows in true hierarchy order (1-3-5-…)
//         instead of grouping merely by depth.                                          */
//     const { rows } = await pg.query(
//       `WITH RECURSIVE t AS (
//          SELECT s.id,
//                 s.name,
//                 r.name         AS role,
//                 s.parent_id,
//                 ARRAY[s.id]    AS path          -- path so far
//            FROM staff s
//            JOIN roles r ON r.id = s.role_id
//           WHERE s.id = $1                       -- root = current staff
//          UNION ALL
//          SELECT s.id,
//                 s.name,
//                 r.name,
//                 s.parent_id,
//                 t.path || s.id                  -- append self to parent's path
//            FROM staff s
//            JOIN t        ON s.parent_id = t.id  -- children
//            JOIN roles r  ON r.id      = s.role_id
//         )
//         SELECT id, name, role, parent_id,
//                array_length(path,1)-1 AS depth, path
//           FROM t
//       ORDER BY path`,                            // ⇐ keeps siblings together
//       [req.staff.id]
//     );

//     res.json({ status: "success", data: rows });
//   } catch (err) { next(err); }
// };
// 5. GET TREE  (staff + players, fully ordered, with parent_id & depth)
exports.getTree = async (req, res, next) => {
  try {
    const { rows } = await pg.query(
      `WITH RECURSIVE staff_tree AS (                               -- ① staff only
         SELECT  s.id,
                 s.name,
                 r.name        AS role,
                 s.parent_id,
                 0             AS depth,
                 ARRAY[s.id]   AS path
           FROM staff s
      INNER JOIN roles r ON r.id = s.role_id
          WHERE s.id = $1                                       -- root = me
         UNION ALL
         SELECT  s.id,
                 s.name,
                 r.name,
                 s.parent_id,
                 st.depth + 1,
                 st.path || s.id
           FROM staff s
      INNER JOIN staff_tree st ON s.parent_id = st.id
      INNER JOIN roles r       ON r.id       = s.role_id
      ),
      full_tree AS (                                             -- ② add players
         SELECT * FROM staff_tree
         UNION ALL
         SELECT  u.id,
                 u.name,
                 'User'              AS role,
                 u.parent_staff_id   AS parent_id,
                 st.depth + 1        AS depth,
                 st.path  || u.id    AS path
           FROM users u
      INNER JOIN staff_tree st ON u.parent_staff_id = st.id
      )
      SELECT id, name, role, parent_id, depth, path             -- ③ final output
        FROM full_tree
    ORDER BY path;`,
      [req.staff.id]
    );

    res.json({ status: "success", data: rows });
  } catch (err) { next(err); }
};
function genSlug (len = 7) {
  return Math.random().toString(36).slice(2, 2 + len);
}
// 6. EDIT STAFF
exports.editStaff = async (req, res, next) => {
  try {
    const { id } = req.params;
    const { name, phone, country, password , percentage} = req.body;   // ← include password

    /* ── hierarchy / privilege checks (unchanged) ────────────────────── */
    await assertDescendant(pg, req.staff.id, id);

    const lvlRes = await pg.query(
      'SELECT role_id FROM staff WHERE id=$1', [id]);
    const lvl = await getLevel(lvlRes.rows[0].role_id);
    if (lvl <= req.staff.level && req.staff.level !== 0)
      return res.status(403).json({ status:'fail', message:'Privilege too low' });

    /* ── build UPDATE dynamically ────────────────────────────────────── */
    const fields = ['name = $1', 'phone = $2', 'country = $3', 'percentage = $4'];
    const values = [name, phone, country,percentage];          //    $1, $2, $3
    let idx      = 4;                               // next placeholder

    if (password && password.trim()) {              // only if sent
      const hash = await bcrypt.hash(password.trim(), 10);
      fields.push(`password = $${idx}`);            // add to SQL
      values.push(hash);                            // add to params
      idx++;
    }

    values.push(id);                                // last $n = id

    const sql = `
      UPDATE staff
         SET ${fields.join(', ')}
       WHERE id = $${values.length}`;               // final placeholder

    await pg.query(sql, values);
 if (phone.trim()) {
     let { rows:[r] } = await pg.query(
  'SELECT slug FROM staff_whatsapp_ref WHERE staff_id = $1',
  [id]
);

const slug = r ? r.slug : genSlug();   // keep old or create new

await pg.query(
  `INSERT INTO staff_whatsapp_ref (staff_id, slug, phone)
     VALUES ($1, $2, $3)
     ON CONFLICT (staff_id)
     DO UPDATE SET phone = EXCLUDED.phone`,
  [id, slug, phone.trim()]
);
    }
    res.json({ status:'success' });
  } catch (err) {
    next(err);
  }
};
// 7. DELETE STAFF
// controllers/staffController.js  (only this handler)

exports.deleteStaff = async (req, res, next) => {
  const leaverId   = Number(req.params.id);   // staff we are deleting
  const operatorId = req.staff.id;            // who calls the endpoint
  const SUPER_ID   = 1;                       // constant root

  if (leaverId === operatorId)
    return res.status(400).json({ status:'fail', message:'Self-delete forbidden' });

                           // using pooled pg already

  await pg.query('BEGIN');
  try {
    /* ───── 1. security checks (unchanged) ───── */
    await assertDescendant(pg, operatorId, leaverId);

    const { rows:[{ role_id }] } =
      await pg.query('SELECT role_id FROM staff WHERE id = $1', [leaverId]);
    const tgtLevel = await getLevel(role_id);
    if (tgtLevel <= req.staff.level && req.staff.level !== 0)
      return res.status(403).json({ status:'fail', message:'Privilege too low' });

    /* ───── 2. collect the whole down-line ───── */
    const { rows:desc } = await pg.query(
      `SELECT descendant_id
         FROM staff_hierarchy
        WHERE ancestor_id = $1
          AND descendant_id <> $1`,           // exclude the leaver itself
      [leaverId]
    );
    const downLine = desc.map(r => Number(r.descendant_id));   // may be []

    /* ───── 3. attach every descendant to Super-Admin ───── */
    if (downLine.length) {
      //                                ↓↓↓ dynamic IN list
      await pg.query(
        `UPDATE staff
            SET parent_id = $1
          WHERE id = ANY($2::bigint[])`,
        [SUPER_ID, downLine]
      );
    }

    /* ───── 4. users that were *directly* attached to the leaver ───── */
    await pg.query(
      `UPDATE users
          SET parent_staff_id = $1
        WHERE parent_staff_id = $2`,
      [SUPER_ID, leaverId]
    );

    /* ───── 5. transfer remaining INR (already in your code) ───── */
    const remain = (
      await pg.query(
        'SELECT inr FROM staff_balances WHERE staff_id = $1', [leaverId])
    ).rows[0]?.inr || 0;
    if (remain > 0)
      await transferInr(pg, 'staff', leaverId,
                               'staff', SUPER_ID, remain, true);

    /* ───── 6. clean up auxiliary tables that reference the leaver ───── */
    await pg.query('DELETE FROM staff_balances WHERE staff_id = $1', [leaverId]);
    await pg.query('DELETE FROM staff_whatsapp_ref WHERE staff_id = $1',  [leaverId]);
    /* add other one-row side-tables here if you have them */

    /* ───── 7. finally remove the staff row itself ───── */
    await pg.query('DELETE FROM staff WHERE id = $1', [leaverId]);

    /* triggers on staff will now:
       – delete the corresponding staff_hierarchy rows for the leaver
       – rebuild paths for every descendant we just re-parented         */

    await pg.query('COMMIT');
    return res.status(204).end();

  } catch (err) {
    await pg.query('ROLLBACK');
    next(err);
  }
};

exports.deletePlayer = async (req, res) => {
  const playerId = Number(req.params.id);
  if (!Number.isFinite(playerId))
    return res.status(400).json({ message:'Bad player id' });

  /* ───── security: may the caller touch this uid? ───── */
  const allowed  = await visibleUserIds(req.staff.id);
  if (!allowed.includes(String(playerId)))
    return res.status(403).json({ message:'Player not in your hierarchy' });

  /* ───── dynamic discovery of referencing tables ───── */
  const possCols = ['user_id','userid','uid','id_user','user'];   // lower-case
  const { rows:tbls } = await pg.query(
    `SELECT table_name, column_name
       FROM information_schema.columns
      WHERE table_schema = 'public'
        AND table_name  <> 'users'
        AND lower(column_name) = ANY($1)`,
    [possCols]
  );

  const targets = tbls.map(r=>({table:r.table_name,col:r.column_name}));

  await pg.query('BEGIN');
  try {
    const log = [];

    for (const {table,col} of targets) {
      const q = `DELETE FROM "${table}" WHERE "${col}" = $1`;
      const { rowCount } = await pg.query(q, [playerId]);
      if (rowCount)
        log.push({ table, column:col, deleted:rowCount });
    }

    /* credits has a fk uid but may have a different column-name */
    const { rowCount:credDel } =
      await pg.query('DELETE FROM credits WHERE uid = $1', [playerId]);
    if (credDel) log.push({ table:'credits', column:'uid', deleted:credDel });

    /* finally remove the user itself */
    const { rows:userRow, rowCount:uCnt } =
      await pg.query('DELETE FROM users WHERE id = $1 RETURNING *', [playerId]);
    if (!uCnt) { await pg.query('ROLLBACK');            // nothing to do?
      return res.status(404).json({ message:'User not found' }); }

    await pg.query('COMMIT');
    res.json({
      message : `Player ${playerId} and related data deleted`,
      details : log,
      user    : userRow[0]
    });
  } catch (err) {
    await pg.query('ROLLBACK').catch(()=>{});
    console.error('deletePlayer error:', err);
    res.status(500).json({ message:'Internal error', error:err.message });
  }
};


// 8. TRANSFER BALANCE (staff→staff OR staff→player)
exports.transfer = async (req, res, next) => {
  await pg.query("BEGIN");
 try {
   const { toType, toId, amount, direction = "deposit" } = req.body;

   if (!["staff","user"].includes(toType))
     return res.status(400).json({ error:"Bad toType" });
   if (!["deposit","withdraw"].includes(direction))
     return res.status(400).json({ error:"Bad direction" });

   /* --- hierarchy ---------------------------------------------------- */
     if (toType === "staff") {
    await assertDescendant(pg, req.staff.id, toId);   // pass the pool/pg
   } else {  // player
     const ok = (await pg.query(
       `SELECT 1
          FROM users u
          JOIN staff_hierarchy h ON h.descendant_id = u.parent_staff_id
         WHERE h.ancestor_id = $1 AND u.id = $2`,
       [req.staff.id, toId])).rowCount;
     if (!ok) return res.status(403).json({ error:"Target not in your tree" });
   }

   /* --- decide src/dst & debit rules -------------------------------- */
   const srcType = direction === "withdraw" ? toType   : "staff";
   const srcId   = direction === "withdraw" ? toId     : req.staff.id;
   const dstType = direction === "withdraw" ? "staff"  : toType;
   const dstId   = direction === "withdraw" ? req.staff.id : toId;

   /* --- balance check (when I am the payer) ------------------------- */
   if (direction === "deposit" && req.staff.level !== 0) {
     /* you are paying -> check *your* balance                            */
     const bal = (
       await pg.query("SELECT inr FROM staff_balances WHERE staff_id=$1",
                      [req.staff.id])
     ).rows[0]?.inr || 0;
     if (amount > bal) return res.status(400).json({ error:"Insufficient INR" });
   }
   if (direction === "withdraw") {
     /* child is paying – ensure the *child* has enough                    */
     if (srcType === "staff") {
       const bal = (
         await pg.query("SELECT inr FROM staff_balances WHERE staff_id=$1",
                        [srcId])
       ).rows[0]?.inr || 0;
       if (amount > bal) return res.status(400).json({ error:"Child has insufficient INR" });
     } else { /* user */ 
       const bal = (
         await pg.query("SELECT inr FROM credits WHERE uid=$1", [srcId])
       ).rows[0]?.inr || 0;
       if (amount > bal) return res.status(400).json({ error:"Player has insufficient INR" });
     }
   }

    await transferInr(
     pg,
     srcType, srcId,
     dstType, dstId,
     amount,
     false,                // skipDebit
     direction             // <── just 'deposit' | 'withdraw'
  );
   await pg.query("COMMIT");
   res.json({ status:"success" });

 } catch(err) { await pg.query("ROLLBACK"); next(err); }
 finally     { }
};

// 9. CHANGE PASSWORD  (self OR SuperAdmin resets anyone)
exports.changePassword = async (req, res, next) => {
  try {
    const { oldPassword, newPassword, targetId } = req.body;
    const self = !targetId || Number(targetId) === req.staff.id;
    const target = self ? req.staff.id : targetId;

    if (!self && req.staff.level !== 0)
      return res.status(403).json({ status:'fail', message:'Only SuperAdmin may reset others' });

    if (self) {
      const { rows:[row] } = await pg.query(
        'SELECT password FROM staff WHERE id=$1', [target]
      );
      if (!(await bcrypt.compare(oldPassword, row.password)))
        return res.status(400).json({ status:'fail', message:'Old password wrong' });
    }

    const hash = await bcrypt.hash(newPassword, 10);
    await pg.query(
      'UPDATE staff SET password=$1, password2=$2 WHERE id=$3',
      [hash, newPassword, target]
    );
    res.json({ status:'success' });
  } catch (err) { next(err); }
};


// in controllers/staffController.js
exports.getById = async (req, res, next) => {
  const id = parseInt(req.params.id, 10);
  if (isNaN(id)) {
    return res.status(400).json({ error: "Invalid staff ID" });
  }

  try {
    const { rows } = await pg.query(
      `SELECT 
         s.id,
         s.name,
         s.email,
         s.phone,
          s.country,
         r.name AS role,
         COALESCE(b.inr, 0) AS balance
       FROM staff s
       JOIN roles r       ON r.id = s.role_id
       LEFT JOIN staff_balances b ON b.staff_id = s.id
       WHERE s.id = $1`,
      [id]
    );

    if (!rows.length) {
      return res.status(404).json({ error: "Staff not found" });
    }

    res.json(rows[0]);
  } catch (err) {
    next(err);
  }
};



/* ────────── helper: IDs visible to <id> (self + descendants) ───────── */
const getTreeIdsSql = `
WITH RECURSIVE tree AS (
  SELECT id FROM staff WHERE id = $1
  UNION ALL
  SELECT s.id
  FROM staff s
  JOIN tree t ON s.parent_id = t.id
)
SELECT array_agg(id) AS ids FROM tree
`;

async function visibleIds(staffId) {
  const { rows:[{ ids }] } = await pg.query(getTreeIdsSql, [staffId]);
  return ids;                   // => e.g. [1,4,9,12]
}

/* ────────── METRICS  (staff_cnt / player_cnt / bank) ──── */

// exports.metrics = async (req, res, next) => {
//   try {
//     const me =parseInt(req.params.id, 10);
//     if (!Number.isInteger(me))
//       return res.status(400).json({ error: "Invalid staff ID" });

//     const ids = await visibleIds(me);            // bigint[]  ≈  [me, ...desc]

//     const { rows:[row] } = await pg.query(
//       `
//       WITH s AS (
//         SELECT COUNT(*) FILTER (WHERE id <> $1) AS staff_cnt
//         FROM   staff
//         WHERE  id = ANY($2)                     -- $2 = bigint[]
//       ),
//       u AS (
//         SELECT COUNT(*) AS player_cnt
//         FROM   users
//         WHERE  parent_staff_id = ANY($2)
//       ),
//       b AS (
//         SELECT COALESCE(SUM(inr),0)::numeric AS bank
//         FROM   staff_balances
//         WHERE  staff_id = ANY($2)
//       )
//       SELECT s.staff_cnt, u.player_cnt, b.bank
//       FROM   s, u, b
//       `,
//       [me, ids]                                  // <-- param list
//     );

//     res.json(row);
//   } catch (err) { next(err); }
// };
// controllers/staffController.js
// … other handlers …

/* ───────────────────────────────────────────────────────── metrics */
exports.metrics = async (req, res, next) => {
  try {
    /* ------------------------------------------------- 1. sanity-check ID */
    const me = Number.parseInt(req.params.id, 10);
    if (!Number.isInteger(me))
      return res.status(400).json({ error: "Invalid staff ID" });

    /* ------------------------------------------------- 2. visible IDs     */
    const ids = await visibleIds(me);          // bigint[]  [me, …descendants]

    /* ------------------------------------------------- 3. aggregate data
       – staff_cnt   → **all** descendant staff (level < User) EXCEPT me
       – player_cnt  → all players whose parent_staff_id is *any* visible ID
       – bank        → **ONLY** the balance that belongs to `me`
    */
    const { rows: [row] } = await pg.query(
      `
      WITH s AS (
        SELECT COUNT(*) AS staff_cnt
        FROM   staff
        WHERE  id <> $1          -- skip self
        AND    id = ANY($2)      -- only descendants in my tree
      ),
      u AS (
        SELECT COUNT(*) AS player_cnt
        FROM   users
        WHERE  parent_staff_id = ANY($2)
      ),
      b AS (
        SELECT COALESCE(inr, 0)::numeric AS bank
        FROM   staff_balances
        WHERE  staff_id = $1      -- <-- ►  only *my* balance
      )
      SELECT s.staff_cnt, u.player_cnt, b.bank
      FROM   s, u, b;
      `,
      [me, ids]                 // $1 = me, $2 = bigint[] (visible IDs)
    );

    /* ------------------------------------------------- 4. respond */
    res.json({ status: "success", ...row });

  } catch (err) { next(err); }
};

/* ────────── ANALYTICS  (7-day trend data) ─────────────── */

exports.analytics = async (req, res, next) => {
  try {
    const staffId =parseInt(req.params.id, 10);
    if (!Number.isInteger(staffId))
      return res.status(400).json({ error: "Invalid staff ID" });

    const ids = await visibleIds(staffId);

    /* staff registrations last 7 days */
    const staffTrendRes = await pg.query(
      `SELECT to_char(created_at::date,'YYYY-MM-DD') AS day,
              COUNT(*) AS count
       FROM staff
       WHERE id      = ANY($1)
         AND created_at >= NOW() - INTERVAL '6 days'  
       GROUP BY day ORDER BY day`,
      [ids]
    );

    /* player registrations (linked by parent_staff_id) last 7 days */
    const playerTrendRes = await pg.query(
      `SELECT to_char(created::date,'YYYY-MM-DD') AS day,
              COUNT(*) AS count
       FROM users
       WHERE parent_staff_id = ANY($1)                 -- ← changed column
         AND created        >= NOW() - INTERVAL '6 days'
       GROUP BY day ORDER BY day`,
      [ids]
    );

    res.json({
      staffTrend:  staffTrendRes.rows,   // [{ day:'2025-06-18', count:3 }, …]
      playerTrend: playerTrendRes.rows
    });
  } catch (err) { next(err); }
};

 const getTreeIds = async (rootId) => {
   const { rows:[{ ids }] } = await pg.query(`
     WITH RECURSIVE tree AS (
       SELECT id FROM staff WHERE id = $1
       UNION ALL
       SELECT s.id
       FROM staff s
       JOIN tree t ON s.parent_id = t.id
     )
     SELECT array_agg(id) AS ids
     FROM tree
   `,[rootId]);
   return ids || [];
 };
  //--------------------------------------------------------------------
 // BULK STATUS PATCH
 //--------------------------------------------------------------------
 /**
  * body: { ids:number[], status:"active"|"inactive" }
  * Only descendants of requester (or SuperAdmin) will be touched.
  */
 exports.bulkStatus = async (req,res,next)=>{
   try{
     const { ids = [], status } = req.body;
     if(!Array.isArray(ids) || !ids.length)
       return next(boom.badRequest("ids[] required"));
     
     if(!["active","inactive"].includes(status))
       return next(boom.badRequest("status must be active|inactive"));
 
     // check hierarchy
     const visible = await getTreeIds(req.user.id);
     const illegal = ids.find(id => !visible.includes(id));
     if(illegal) return next(boom.forbidden("One or more IDs outside your tree"));
 
     await pg.query(`
       UPDATE staff SET status = $2 WHERE id = ANY($1)
     `,[ids,status]);
 
     res.json({ updated: ids.length, status });
   }catch(err){ next(err); }
 };

 // controllers/staffController.js  (add at bottom)
exports.transferHistory = async (req,res,next)=>{
  try{
    // ► all accounts I’m allowed to see (self + descendants)
    const ids = await visibleIds(req.staff.id);

    /* both directions: money I sent OR received */
    const { rows } = await pg.query(
      `SELECT id,
              from_type, from_id,
              to_type,   to_id,
              amount,
              created_at
         FROM staff_transfers
        WHERE (from_type='staff' AND from_id = ANY($1))
           OR (to_type  ='staff' AND to_id   = ANY($1))
     ORDER BY created_at DESC
     LIMIT 200`,                           
      [ids]
    );

    res.json({ status:'success', data:rows });
  }catch(err){ next(err); }
};
/* ───────────────────────────────────────────────────── ROLL-UPS */
/**
 * Helper that returns
 *   { staff_id, tree_balance, own_balance }
 * for every id in `ids`.
 */
async function fetchRollups(pg, ids) {
  const { rows } = await pg.query(
    `SELECT staff_id, tree_balance, own_balance
       FROM v_staff_rollup
      WHERE staff_id = ANY($1::bigint[])`,
    [ids]
  );
  return rows;
}


/* ───────────────── roll-up for every visible UID ─────────────── */
exports.rollupAll = async (req, res, next) => {
  try {
     const id  = Number(req.params.id);
    /* 1️⃣  all IDs that I’m allowed to see (me + descendants) */
    const ids = await visibleIds(id);          // bigint[]
    // console.log("rollupAll: visible IDs", ids);

    /* 2️⃣  single-scan aggregate */
    const { rows } = await pg.query(`
      WITH balances AS (
        SELECT staff_id, inr
        FROM   staff_balances
        WHERE  staff_id = ANY($1)
      ),
      tree AS (
        SELECT ancestor_id, descendant_id
        FROM   staff_hierarchy
        WHERE  ancestor_id = ANY($1)
      )
      SELECT b1.staff_id,
             b1.inr                               AS own_balance,
             COALESCE(SUM(b2.inr), 0)             AS downline_balance
      FROM   balances  b1
      LEFT   JOIN tree       t  ON t.ancestor_id = b1.staff_id
      LEFT   JOIN balances  b2  ON b2.staff_id   = t.descendant_id
      GROUP  BY b1.staff_id, b1.inr              -- ✱ fixed line ✱
      ORDER  BY b1.staff_id;
    `, [ids]);

    res.json({ status: "success", data: rows });

  } catch (e) { next(e); }
};


/* ───────────────── roll-up for ONE UID ────────────────────────── */
exports.rollupById = async (req,res,next)=>{
  try{
    const id  = Number(req.params.id);
    if(Number.isNaN(id)) return res.status(400).json({error:"Bad id"});

    /* same query but restricted */
    const { rows:[row] } = await pg.query(
      `SELECT sb.staff_id                                   AS staff_id,
              sb.inr                                        AS own_balance,
              COALESCE((
                SELECT SUM(inr)
                  FROM staff_balances
                  WHERE staff_id <> sb.staff_id
                    AND staff_id = ANY($2)
              ),0)                                          AS downline_balance
         FROM staff_balances sb
        WHERE sb.staff_id=$1`,
      [id, await visibleIds(id)]
    );

    res.json({ status:"success", data:row });
  }catch(e){ next(e); }
};

/* ───────────────────────────────────────────────────────────
   GET /api/staff/transfers?dir=deposit|withdraw
   --------------------------------------------------------- */
/* ───────────────────────────────────────────────────────────
   GET /api/staff/transfers[/:id]?dir=deposit|withdraw&page&limit
   --------------------------------------------------------- */
// exports.listTransfers = async (req, res, next) => {
//   try {
//     /* 1. which subtree? */
//     const root = req.params.id ? Number(req.params.id) : Number(req.staff.id);
//     if (!Number.isInteger(root))
//       return res.status(400).json({ error: "Invalid staff ID" });

//     /* 2. security – compare numbers, not strings */
//     const myTree = (await visibleIds(req.staff.id)).map(Number); // 👈 cast!
//     if (!myTree.includes(root))
//       return res.status(403).json({ error: "Not in your hierarchy" });

//     const tree = myTree.includes(root)
//                ? myTree                       // we already have numbers
//                : (await visibleIds(root)).map(Number);
//     /* 3. optional dir filter */
//     const dir = (req.query.dir || "").toLowerCase();
//     if (dir && !["deposit", "withdraw"].includes(dir))
//       return res.status(400).json({ error: "dir must be deposit|withdraw" });

//     /* 4. pagination */
//     const page  = Math.max(1, +(req.query.page  || 1));
//     const limit = Math.min(100, +(req.query.limit || 50));
//     const off   = (page - 1) * limit;

//     /* 5. query with names + correct dir filter */
//     const { rows } = await pg.query(
//       `WITH x AS (
//   SELECT t.id, t.from_type, t.from_id,
//          t.to_type,   t.to_id,
//          t.amount,    t.created_at,
//          CASE
//            WHEN t.from_type = 'staff' AND t.from_id = $2 THEN 'deposit'
//            WHEN t.to_type   = 'staff' AND t.to_id   = $2 THEN 'withdraw'
//            ELSE 'internal'
//          END                                              AS direction
//     FROM staff_transfers t
//    WHERE (t.from_type = 'staff' AND t.from_id = ANY($1))
//       OR (t.to_type   = 'staff' AND t.to_id   = ANY($1))
// )
//         SELECT  x.*,
//                 COALESCE(sf_from.name, u_from.name) AS from_name,
//                 COALESCE(sf_to.name,  u_to.name )   AS to_name
//           FROM  x
//      LEFT JOIN staff sf_from ON x.from_type = 'staff' AND sf_from.id = x.from_id
//      LEFT JOIN staff sf_to   ON x.to_type   = 'staff' AND sf_to.id   = x.to_id
//      LEFT JOIN users u_from  ON x.from_type = 'user'  AND u_from.id  = x.from_id
//      LEFT JOIN users u_to    ON x.to_type   = 'user'  AND u_to.id    = x.to_id
//          ${dir ? "WHERE x.direction = $2" : ""}
//      ORDER BY x.created_at DESC
//      OFFSET $${dir ? 3 : 2} LIMIT $${dir ? 4 : 3}`,
//       dir ? [tree, dir, off, limit] : [tree, off, limit]
//     );

//     res.json({ status: "success", data: rows, page, limit });
//   } catch (e) { next(e); }
// };

/* ------------------------------------------------------------------------
   GET /api/staff/transfers[/:id]?dir=deposit|withdraw&page&limit
   ------------------------------------------------------------------------ */
/* ------------------------------------------------------------------
   GET /api/staff/transfers[/:id]
        ?dir=deposit|withdraw&page&limit
------------------------------------------------------------------- */
exports.listTransfers = async (req, res, next) => {
  try {
    /* 1 ─ pick subtree root */
    const root = req.params.id ? Number(req.params.id) : Number(req.staff.id);
    if (!Number.isInteger(root))
      return res.status(400).json({ error: "Invalid staff ID" });

    /* 2 ─ security */
    const myTree = (await visibleIds(req.staff.id)).map(Number);
    if (!myTree.includes(root))
      return res.status(403).json({ error: "Not in your hierarchy" });

    const tree = await visibleIds(root);              // bigint[]

    /* 3 ─ filters */
    const dir = (req.query.dir || "").toLowerCase();
    if (dir && !["deposit", "withdraw"].includes(dir))
      return res.status(400).json({ error: "dir must be deposit|withdraw" });

    const page  = Math.max(1, +(req.query.page  || 1));
    const limit = Math.min(100, +(req.query.limit || 50));
    const off   = (page - 1) * limit;

    /* 4 ─ main query */
    const { rows } = await pg.query(
`WITH scoped AS (
   SELECT t.id, t.from_type, t.from_id,
          t.to_type,   t.to_id,
          t.amount,    t.created_at,
          CASE
            WHEN t.from_type = 'staff' AND t.from_id = $2 THEN 'deposit'
            WHEN t.to_type   = 'staff' AND t.to_id   = $2 THEN 'withdraw'
            ELSE 'internal'
          END AS dir
     FROM staff_transfers t
    WHERE (t.from_type = 'staff' AND t.from_id = ANY($1))
       OR (t.to_type   = 'staff' AND t.to_id   = ANY($1))
)
SELECT  s.id,
        s.from_type, s.from_id,
        s.to_type,   s.to_id,
        s.amount,    s.created_at,
        s.dir        AS direction,                    -- ← exposed name
        COALESCE(sf_from.name, u_from.name) AS from_name,
        COALESCE(sf_to.name,   u_to.name)   AS to_name
  FROM  scoped s
  LEFT JOIN staff sf_from ON s.from_type = 'staff' AND sf_from.id = s.from_id
  LEFT JOIN staff sf_to   ON s.to_type   = 'staff' AND sf_to.id   = s.to_id
  LEFT JOIN users u_from  ON s.from_type = 'user'  AND u_from.id  = s.from_id
  LEFT JOIN users u_to    ON s.to_type   = 'user'  AND u_to.id    = s.to_id
  ${dir ? "WHERE s.dir = $3" : ""}
  ORDER BY s.created_at DESC
  OFFSET $${dir ? 4 : 3} LIMIT $${dir ? 5 : 4}`,
      dir
        ? [tree, root, dir, off, limit]   // $1 $2 $3 $4 $5
        : [tree, root,      off, limit]   // $1 $2      $3 $4
    );

    res.json({ status: "success", page, limit, data: rows });
  } catch (err) { next(err); }
};


/* ───────────────────────────────────────────────────────────
   GET /api/staff/transfers/summary
   --------------------------------------------------------- */
exports.transfersSummary = async (req, res, next) => {
  try {
    const subtree = await visibleIds(req.staff.id);

    const { rows:[row] } = await pg.query(
      `SELECT
         SUM( CASE WHEN from_type='staff' AND from_id = ANY($1) THEN amount ELSE 0 END ) AS total_deposit,
         SUM( CASE WHEN to_type  ='staff' AND to_id   = ANY($1) THEN amount ELSE 0 END ) AS total_withdraw
       FROM staff_transfers
       WHERE (from_type='staff' AND from_id = ANY($1))
          OR (to_type  ='staff' AND to_id   = ANY($1))`,
      [subtree]
    );

    res.json({ status:"success", ...row });
  } catch (e) { next(e); }
};
/* ───────────────────────────────────────────────────────────
   GET /api/staff/transfers/summary/:id
   --------------------------------------------------------- */
exports.transfersSummaryById = async (req, res, next) => {
  try {
    const target = Number(req.params.id);
    if (!Number.isInteger(target))
      return res.status(400).json({ error:"Invalid staff ID" });

    /* security: target must be in *my* subtree */
    const subtree = await visibleIds(req.staff.id);
    if (!subtree.includes(target))
      return res.status(403).json({ error:"Not in your hierarchy" });

    const theirTree = await visibleIds(target);

    const { rows:[row] } = await pg.query(
      `SELECT
         SUM( CASE WHEN from_type='staff' AND from_id = ANY($1) THEN amount ELSE 0 END ) AS total_deposit,
         SUM( CASE WHEN to_type  ='staff' AND to_id   = ANY($1) THEN amount ELSE 0 END ) AS total_withdraw
       FROM staff_transfers
       WHERE (from_type='staff' AND from_id = ANY($1))
          OR (to_type  ='staff' AND to_id   = ANY($1))`,
      [theirTree]
    );

    res.json({ status:"success", ...row });
  } catch (e) { next(e); }
};
exports.getWhatsappRef = async (req, res, next) => {
  try {
    const { rows:[r] } = await pg.query(
      'SELECT slug, phone FROM staff_whatsapp_ref WHERE staff_id = $1',
      [req.params.id]
    );
    if (!r) return res.status(404).json({ status:'fail', message:'No ref' });
    res.json({
      slug  : r.slug,
      phone : r.phone,
      link  : `https://api.boss707.com/?ref=${r.slug}`
    });
  } catch (e) { next(e); }
};


const cols = `
  s.id,
  s.name,
  COALESCE(s.percentage, 0)::numeric AS percentage,
  r.name  AS role,
  r.level AS level,
  s.parent_id
`;

/* ---- upstream chain ---------------------------------------------------- */
exports.percentChain = async (req, res, next) => {
  const viewer = req.staff.id;
  const id     = req.params.id;

 
  try {
    await assertDescendant(pg, viewer, id);

    const { rows } = await pg.query(`
      WITH RECURSIVE chain AS (
        SELECT ${cols}, 0 depth
          FROM staff s JOIN roles r ON r.id = s.role_id
         WHERE s.id = $1
        UNION ALL
        SELECT ${cols}, depth+1
          FROM staff s
          JOIN chain c ON c.parent_id = s.id
          JOIN roles r ON r.id = s.role_id
      )
      SELECT * FROM chain ORDER BY depth DESC;
    `, [id]);

    res.json(rows);
  } catch (e) { next(e); }
  finally    { }
};

/* ---- full subtree ------------------------------------------------------ */
exports.percentTree = async (req, res, next) => {
  const viewer = req.staff.id;
  const rootId = req.params.id;

 
  try {
    await assertDescendant(pg, viewer, rootId);

    const { rows } = await pg.query(`
      WITH RECURSIVE tree AS (
        SELECT ${cols}, 0 depth
          FROM staff s JOIN roles r ON r.id = s.role_id
         WHERE s.id = $1
        UNION ALL
        SELECT ${cols}, depth+1
          FROM staff s
          JOIN tree t ON s.parent_id = t.id
          JOIN roles r ON r.id = s.role_id
      )
      SELECT * FROM tree ORDER BY depth, id;
    `, [rootId]);

    res.json(rows);
  } catch (e) { next(e); }
  finally    { }
};