const express = require('express');
const router = express.Router();
const pg = require('../General/Model');

// Get user withdrawals with pagination
router.get('/user/withdrawals/:userId', async (req, res) => {
    try {
        const userId = req.params.userId;
        const {
            page = 1,
            limit = 10,
            status,
            startDate,
            endDate,
            coin
        } = req.query;

        // Build count query with filters
        let countQuery = `
            SELECT COUNT(*) 
            FROM withdrawals w
            WHERE w.uid = $1
        `;
        
        const values = [userId];
        let valueIndex = 2;

        if (status) {
            countQuery += ` AND w.status = $${valueIndex}`;
            values.push(status);
            valueIndex++;
        }

        if (startDate) {
            countQuery += ` AND w.date >= $${valueIndex}`;
            values.push(startDate);
            valueIndex++;
        }

        if (endDate) {
            countQuery += ` AND w.date <= $${valueIndex}`;
            values.push(endDate);
            valueIndex++;
        }

        if (coin) {
            countQuery += ` AND w.coin = $${valueIndex}`;
            values.push(coin);
            valueIndex++;
        }

        const countResult = await pg.query(countQuery, values);
        const totalItems = parseInt(countResult.rows[0].count);

        // Build main query with filters
        let query = `
            SELECT 
                w.uid,
                w.date,
                w.amount,
                w.wallet,
                w.coin,
                w.status
            FROM withdrawals w
            WHERE w.uid = $1
        `;

        if (status) query += ` AND w.status = $2`;
        if (startDate) query += ` AND w.date >= $3`;
        if (endDate) query += ` AND w.date <= $4`;
        if (coin) query += ` AND w.coin = $5`;

        const offset = (parseInt(page) - 1) * parseInt(limit);
        query += ` ORDER BY w.date DESC LIMIT $${valueIndex} OFFSET $${valueIndex + 1}`;
        values.push(parseInt(limit), offset);

        const result = await pg.query(query, values);

        // Convert amounts to USDT
        const withdrawals = await Promise.all(result.rows.map(async (withdraw) => {
            const amountInUSDT = await convertToUSDT(parseFloat(withdraw.amount), withdraw.coin);
            return { ...withdraw, amount_usdt: amountInUSDT };
        }));

        res.json({
            success: true,
            data: withdrawals,
            page: parseInt(page),
            limit: parseInt(limit),
            total: totalItems,
            totalPages: Math.ceil(totalItems / parseInt(limit))
        });
    } catch (error) {
        console.error('Error fetching user withdrawals:', error);
        res.status(500).json({
            success: false,
            error: 'Internal server error'
        });
    }
});

// Get user withdrawal stats
router.get('/user/withdrawals/stats/:userId', async (req, res) => {
    try {
        const userId = req.params.userId;
        
        const stats = await pg.query(`
            SELECT 
                COUNT(*)::integer as total_transactions,
                COALESCE(SUM(amount), 0)::float as total_amount,
                COUNT(CASE WHEN LOWER(status) = 'completed' THEN 1 END)::integer as completed_transactions,
                COUNT(CASE WHEN LOWER(status) = 'pending' THEN 1 END)::integer as pending_transactions,
                COUNT(CASE WHEN LOWER(status) = 'failed' THEN 1 END)::integer as failed_transactions,
                array_agg(DISTINCT coin) as coins_used
            FROM withdrawals
            WHERE uid = $1
        `, [userId]);

        // Calculate total amount in USDT for each coin
        const withdrawalsByCoin = await pg.query(`
            SELECT coin, SUM(amount) as total_amount
            FROM withdrawals
            WHERE uid = $1
            GROUP BY coin
        `, [userId]);

        const totalUSDT = await Promise.all(
            withdrawalsByCoin.rows.map(async (item) => {
                const amountInUSDT = await convertToUSDT(parseFloat(item.total_amount), item.coin);
                return amountInUSDT;
            })
        );

        const totalAmountUSDT = totalUSDT.reduce((acc, curr) => acc + curr, 0);

        res.json({
            success: true,
            data: {
                ...stats.rows[0],
                total_amount_usdt: totalAmountUSDT
            }
        });
    } catch (error) {
        console.error('Error fetching user withdrawal stats:', error);
        res.status(500).json({
            success: false,
            error: 'Internal server error'
        });
    }
});

module.exports = router;