const express = require('express');
const router = express.Router();
const gameController = require('./controller');
const { check, query } = require('express-validator');
const db = require('../General/Model');
const NodeCache = require('node-cache');


// Route for paginated games by vendor
router.get(
  '/by-vendor',
  gameController.constructor.getValidationRules(),
  gameController.getGamesByVendor
);

// Route for vendors with game counts
router.get('/vendors', 
    [
      query('vendor').optional(),
      query('pageSize').optional().isInt({ min: 1 }).withMessage('Page size must be a positive integer')
    ], 
    gameController.getVendors
  );



  // Initialize cache with 5 minute TTL
const cache = new NodeCache({ stdTTL: 300 });

/**
 * Search API endpoint focusing on title, vendor, and type
 * @route GET /api/games/search
 * @param {string} keyword - Search keyword (supports partial words)
 * @param {number} page - Page number (default: 1)
 * @param {number} limit - Items per page (default: 20)
 */
router.get('/games/search', async (req, res) => {
    try {
        let {
            keyword = '',
            page = 1,
            limit = 20
        } = req.query;

        // Clean and prepare the search keyword
        keyword = keyword.trim().toLowerCase();
        
        // Validate pagination parameters
        const pageNum = Math.max(1, parseInt(page));
        const limitNum = Math.min(50, Math.max(1, parseInt(limit)));
        const offset = (pageNum - 1) * limitNum;

        // Generate cache key
        const cacheKey = `games_search_${keyword}_${pageNum}_${limitNum}`;
        
        // Check cache
        const cachedResult = cache.get(cacheKey);
        if (cachedResult) {
            return res.json({
                success: true,
                data: cachedResult,
                cached: true
            });
        }

        // Focused search query on title, vendor, and type
        const searchQuery = `
            WITH search_results AS (
                SELECT 
                    id,
                    title,
                    platform,
                    type,
                    subtype,
                    vendor,
                    created_at,
                    details_description_en,
                    details_thumbnails_300x300,
                    CASE
                        -- Title matches (highest priority)
                        WHEN title ILIKE $1 THEN 100  -- Exact start match
                        WHEN title ILIKE $2 THEN 80   -- Contains match
                        
                        -- Vendor matches (medium priority)
                        WHEN vendor ILIKE $1 THEN 70  -- Exact start match
                        WHEN vendor ILIKE $2 THEN 60  -- Contains match
                        
                        -- Type matches (lower priority)
                        WHEN type ILIKE $1 THEN 50    -- Exact start match
                        WHEN type ILIKE $2 THEN 40    -- Contains match
                        
                        ELSE 0
                    END as relevance_score
                FROM apigames
                WHERE 
                    title ILIKE $2 OR
                    vendor ILIKE $2 OR
                    type ILIKE $2
            )
            SELECT 
                id,
                title,
                platform,
                type,
                subtype,
                vendor,
                created_at,
                details_description_en,
                details_thumbnails_300x300
            FROM search_results
            WHERE relevance_score > 0
            ORDER BY relevance_score DESC, title ASC
            LIMIT $3 OFFSET $4
        `;

        // Count query for total results
        const countQuery = `
            SELECT COUNT(*) as total
            FROM apigames
            WHERE 
                title ILIKE $1 OR
                vendor ILIKE $1 OR
                type ILIKE $1
        `;

        // Execute queries
        const [searchResult, countResult] = await Promise.all([
            db.query(searchQuery, [
                `${keyword}%`,      // Exact start match
                `%${keyword}%`,     // Partial match
                limitNum,
                offset
            ]),
            db.query(countQuery, [`%${keyword}%`])
        ]);

        const totalItems = parseInt(countResult.rows[0].total);
        const totalPages = Math.ceil(totalItems / limitNum);

        const response = {
            items: searchResult.rows,
            pagination: {
                total: totalItems,
                page: pageNum,
                limit: limitNum,
                totalPages
            },
            meta: {
                query: keyword,
                searchFields: ['title', 'vendor', 'type']
            }
        };

        // Cache the results
        cache.set(cacheKey, response);

        res.json({
            success: true,
            data: response,
            cached: false
        });

    } catch (error) {
        console.error('Search API Error:', error);
        res.status(500).json({
            success: false,
            message: 'An error occurred while searching games',
            error: process.env.NODE_ENV === 'development' ? error.message : undefined
        });
    }
});


module.exports = router;