import React, { Component } from "react";
import styled from "styled-components";
import { Play } from "lucide-react";
import { FontAwesomeIcon } from "@fortawesome/react-fontawesome";
import { faArrowLeft, faSpinner } from "@fortawesome/free-solid-svg-icons";
import { getUID } from "../../../Helper";

const GameOverlay = styled.div`
  position: absolute;
  top: 0;
  left: 0;
  width: 100%;
  height: 100%;
  background-color: rgba(0, 0, 0, 0.4);
  backdrop-filter: blur(5px);
  border-radius: 8px;
  display: flex;
  justify-content: center;
  align-items: center;
  opacity: 0;
  transition: opacity 0.2s ease;
`;

const PlayIconContainer = styled.div`
  width: 50px;
  height: 50px;
  border-radius: 50%;
  background-color: rgba(255, 255, 255, 0.3);
  display: flex;
  justify-content: center;
  align-items: center;
`;

const FullScreenContainer = styled.div`
  position: fixed;
  top: 0;
  left: 0;
  width: 100%;
  height: 100%;
  z-index: 999;
  background-color: #1b2336;
`;

const FullScreenIframe = styled.iframe`
  width: 100%;
  height: 100%;
  border: none;
  display: ${(props) => (props.loading ? "none" : "block")};
`;

const PlayIcon = styled(Play)`
  color: #fff;
  font-size: 24px;
`;

const BackButton = styled.button`
  position: absolute;
  top: 10px;
  left: 10px;
  z-index: 1000;
  background: none;
  border: none;
  color: #fff;
  font-size: 24px;
  cursor: pointer;
  padding: 8px;

  &:hover {
    opacity: 0.8;
  }
`;

const GridWrapper = styled.div`
  display: grid;
  gap: 12px;
  padding: 12px;
  width: 100%;

  /* Mobile - 3 items per row */
  grid-template-columns: repeat(3, 1fr);

  /* Tablet - 4 items per row */
  @media (min-width: 640px) {
    grid-template-columns: repeat(4, 1fr);
  }

  /* Desktop - 6 items per row */
  @media (min-width: 1024px) {
    grid-template-columns: repeat(6, 1fr);
    gap: 16px;
    padding: 16px;
  }

  /* Large Desktop - 8 items per row */
  @media (min-width: 1280px) {
    grid-template-columns: repeat(8, 1fr);
    gap: 20px;
    padding: 20px;
  }x
`;

const LoadMoreButton = styled.button`
  background-color: #03030F;
  color: white;
  border: none;
  padding: 12px 24px;
  border-radius: 8px;
  cursor: pointer;
  font-weight: bold;
  margin: 20px auto;
  display: block;
  transition: background-color 0.3s ease;

  &:hover {
    background-color: #4a5354;
  }

  &:disabled {
    background-color: #ccc;
    cursor: not-allowed;
  }
`;

const LoaderContainer = styled.div`
  display: flex;
  justify-content: center;
  align-items: center;
  padding: 20px;
`;

const Spinner = styled.div`
  width: 40px;
  height: 40px;
  border: 4px solid #1c1e22;
  border-top: 4px solid rgb(18, 20, 24);
  border-radius: 50%;
  animation: spin 1s linear infinite;

  @keyframes spin {
    0% {
      transform: rotate(0deg);
    }
    100% {
      transform: rotate(360deg);
    }
  }
`;

const LotteryGameCard = styled.div`
  width: 100%;
  aspect-ratio: 1.32;
  border-radius: 8px;
  position: relative;
  cursor: pointer;
  display: flex;
  flex-direction: column;
  justify-content: center;
  align-items: center;
  transition: transform 0.2s ease;

  &:hover {
    ${GameOverlay} {
      opacity: 1;
    }
    transform: translateY(-5px);
  }
`;

const LotteryGameImage = styled.img`
  width: 100%;
  height: 100%;
  object-fit: cover;
  border-radius: 8px;
`;

class Lottery extends Component {
  constructor(props) {
    super(props);
    this.state = {
      isLogged: localStorage.getItem("logged") === "true",
      displayGamesLotteryXgaming: [],
      message: null,
      iframeUrl: "",
      showIframe: false,
      iframeLoading: false,
      isLoadingMoreLottery: false,
      lotteryPage: 1,
    };
  }

  async componentDidMount() {
    await this.fetchInitialLotteryGames();
  }

  fetchVendorData = async ({ vendorName, page = 1, perPage = 18 }) => {
    const baseUrl = "https://api.boss707.com/xGaming/by-vendor";
    const vendorCategories = {
      lottery: ["mascot", "caleta", "belatragames", "superlottotv", "lottoinstantwin"],
    };

    const createUrl = (vendor) =>
      `${baseUrl}?vendor=${vendor}&page=${page}&per_page=${perPage}`;

    const fetchSingleVendor = async (vendor) => {
      try {
        const token = localStorage.getItem('token');
        const response = await fetch(createUrl(vendor), {
          headers: {
            'Authorization': `Bearer ${token}`,
          },
        });
        if (!response.ok) throw new Error(`HTTP error! status: ${response.status}`);
        return { vendor, data: await response.json() };
      } catch (error) {
        console.error(`Error fetching data for vendor ${vendor}:`, error);
        return { vendor, error: error.message };
      }
    };

    if (vendorCategories[vendorName]) {
      const vendors = vendorCategories[vendorName];
      const results = [];
      for (const vendor of vendors) {
        const result = await fetchSingleVendor(vendor);
        results.push(result);
      }
      return { category: vendorName, results };
    }

    const result = await fetchSingleVendor(vendorName);
    return { category: "single", results: [result] };
  };

  fetchInitialLotteryGames = async () => {
    try {
      const response = await this.fetchVendorData({
        vendorName: "lottery",
        page: 1,
        perPage: 9, // Match this with loadMore to ensure consistency
      });

      const lotteryGames = response?.results.reduce((games, vendorData) => {
        if (vendorData?.data?.data && Array.isArray(vendorData.data.data)) {
          const vendorGames = vendorData.data.data.map((game) => ({
            id: game.id,
            title: game.title,
            platform: game.platform,
            type: game.type,
            subtype: game.subtype,
            vendor: game.vendor,
            details_description_en: game.details_description_en,
            details_thumbnails_300x300: game.details_thumbnails_300x300,
          }));
          return [...games, ...vendorGames];
        }
        return games;
      }, []);

      this.setState({
        displayGamesLotteryXgaming: lotteryGames,
        lotteryPage: 1,
      });
    } catch (error) {
      console.error("Error fetching initial lottery games:", error);
      this.setState({
        message: { type: "error", text: "Failed to load games. Please try again." },
      });
    }
  };

  loadMoreLotteryGames = async () => {
    this.setState({ isLoadingMoreLottery: true });
    const nextPage = this.state.lotteryPage + 1;

    try {
      const response = await this.fetchVendorData({
        vendorName: "lottery",
        page: nextPage,
        perPage: 18, // Consistent with initial fetch
      });

      const newLotteryGames = response?.results.reduce((games, vendorData) => {
        if (vendorData?.data?.data && Array.isArray(vendorData.data.data)) {
          const vendorGames = vendorData.data.data.map((game) => ({
            id: game.id,
            title: game.title,
            platform: game.platform,
            type: game.type,
            subtype: game.subtype,
            vendor: game.vendor,
            details_description_en: game.details_description_en,
            details_thumbnails_300x300: game.details_thumbnails_300x300,
          }));
          return [...games, ...vendorGames];
        }
        return games;
      }, []);

      this.setState((prevState) => ({
        displayGamesLotteryXgaming: [...prevState.displayGamesLotteryXgaming, ...newLotteryGames],
        lotteryPage: nextPage,
        isLoadingMoreLottery: false,
      }));
    } catch (error) {
      console.error("Error loading more lottery games:", error);
      this.setState({
        isLoadingMoreLottery: false,
        message: { type: "error", text: "Failed to load more games. Please try again." },
      });
    }
  };

  handleIframeLoad = () => {
    this.setState({ iframeLoading: false });
  };

  handleBackButton = () => {
    this.setState({ showIframe: false, iframeUrl: "", iframeLoading: true });
  };

  handleGameLaunchXGaming = async (game) => {
    if (!this.state.isLogged) {
      this.setState({ message: { type: "error", text: "Please login to play live game" } });
      return;
    }
    const token = localStorage.getItem('token');
    try {
      const response = await fetch("https://api.boss707.com/api/casino/gamerun", {
        method: "POST",
        headers: { "Content-Type": "application/json", "Authorization": `Bearer ${token}` },
        body: JSON.stringify({
          game_id: game.id,
          currency: "USD",
          mode: "real",
          language: "en",
          home_url: "https://api.boss707.com",
          device: "desktop",
          user_id: getUID,
          vendor: game.vendor,
          title: game.title,
          coin: "usdt",
        }),
      });

      const result = await response.json();
      if (result.gameRunUrl) {
        this.setState({ iframeUrl: result.gameRunUrl, showIframe: true, iframeLoading: true });
      } else {
        throw new Error("Failed to launch game");
      }
    } catch (error) {
      this.setState({ message: { type: "error", text: error.message } });
    }
  };

  render() {
    const { displayGamesLotteryXgaming, showIframe, iframeUrl, iframeLoading, isLoadingMoreLottery } =
      this.state;

    const imageStyle = {
      position: "absolute",
      top: "50%",
      left: "50%",
      transform: "translate(-50%, -50%)",
      width: "250px",
      animation: "fadeinout 2s infinite",
    };

    const animationStyle = `
      @keyframes fadeinout {
        0%, 100% { opacity: 0; }
        50% { opacity: 1; }
      }
    `;

    if (showIframe) {
      return (
        <FullScreenContainer>
          <style>{animationStyle}</style>
          {iframeLoading && <img src="assets/images/./assets/boss707.png" alt="Loading" style={imageStyle} />}
          <FullScreenIframe
            src={iframeUrl}
            title="Game Iframe"
            onLoad={this.handleIframeLoad}
            loading={iframeLoading}
          />
          <BackButton onClick={this.handleBackButton}>
            <FontAwesomeIcon icon={faArrowLeft} />
          </BackButton>
        </FullScreenContainer>
      );
    }

    return (
      <>
        <GridWrapper>
          {displayGamesLotteryXgaming?.map((game, index) => (
            <LotteryGameCard key={game.id} onClick={() => this.handleGameLaunchXGaming(game)}>
              <LotteryGameImage src={game.details_thumbnails_300x300} alt={`Game ${index}`} />
              <GameOverlay>
                <PlayIconContainer>
                  <PlayIcon />
                </PlayIconContainer>
              </GameOverlay>
            </LotteryGameCard>
          ))}
        </GridWrapper>
        {isLoadingMoreLottery ? (
          <LoaderContainer>
            <Spinner />
          </LoaderContainer>
        ) : (
          <LoadMoreButton onClick={this.loadMoreLotteryGames} disabled={isLoadingMoreLottery}>
            Load More
          </LoadMoreButton>
        )}
      </>
    );
  }
}

export default Lottery;