import React from 'react';
import styled from 'styled-components';
import { Heart, X } from 'lucide-react';
import { getUID } from '../../../../../Helper';

const StyledModal = styled.div`
  position: fixed;
  top: 0;
  left: 0;
  width: 100%;
  height: 100%;
  background-color: rgba(0, 0, 0, 0.5);
  display: ${props => props.show ? 'flex' : 'none'};
  justify-content: center;
  align-items: center;
  z-index: 1000;
`;

const ModalContent = styled.div`
  background-color: #17181B;
  border-radius: 15px;
  width: 90%;
  max-width: 500px;
  color: white;
  position: relative;

  @media (min-width: 576px) {
    width: 350px;
  }
  @media (min-width: 768px) {
    width: 450px;
  }
  @media (min-width: 992px) {
    width: 500px;
  }
`;

const ModalHeader = styled.div`
  padding: 15px;
  display: flex;
  justify-content: space-between;
  align-items: center;
  
  .title {
    font-size: 14px;
    font-weight: bold;
  }

  .close-button {
    background: none;
    border: none;
    color: white;
    cursor: pointer;
    padding: 6px;
    display: flex;
    align-items: center;
    justify-content: center;

    &:hover {
      opacity: 0.8;
    }
  }
`;

const ModalBody = styled.div`
  padding: 0 15px 15px;
`;

const ProfileSection = styled.div`
  text-align: center;
  margin-bottom: 15px;
  padding: 15px 0;

  img {
    width: 80px;
    height: 80px;
    border-radius: 50%;
    margin-bottom: 8px;
  }

  h3 {
    margin-bottom: 8px;
  }

  .level-indicator {
    display: flex;
    justify-content: center;
    gap: 8px;
    align-items: center;
    color: #ff6b6b;
  }
`;

const StatisticsSection = styled.div`
  background-color: #1E2024;
  padding: 15px;
  border-radius: 12px;
  margin-bottom: 15px;

  h4 {
    margin-bottom: 12px;
  }
`;

const StatsGrid = styled.div`
  display: grid;
  grid-template-columns: 1fr 1fr 1fr;
  gap: 10px;
`;

const StatBox = styled.div`
  background-color: #1B1C1F;
  padding: 10px;
  border-radius: 12px;
  text-align: center;
  
  .stat-label {
    color: #676D76;
    font-size: 11px;
    margin-bottom: 3px;
  }
  
  .stat-value {
    font-size: 12px;
    font-weight: bold;
  }
`;

const MedalsSection = styled.div`
  background-color: #1E2024;
  padding: 20px;
  border-radius: 15px;

  h4 {
    margin-bottom: 15px;
  }

  .medals-container {
    display: flex;
    justify-content: center;
    gap: 10px;
    flex-wrap: wrap;
  }

  img {
    width: 44px;
    height: 63px;
  }
`;


const UserInfoModal = ({ show, onClose, userData }) => {
  const [stats, setStats] = useState({ betCount: null, winCount: null });
  const [loading, setLoading] = useState(false);
  const [error, setError] = useState(null);

  const getAuthToken = () => {
    return localStorage.getItem('token');
  };

  useEffect(() => {
    if (!show) return;

    const fetchBetWinCount = async () => {
      setLoading(true);
      setError(null);

      try {
        const token = localStorage.getItem('token');
        const response = await fetch(`https://api.boss707.com/betHistory/user/${getUID()}/bet-win-count`, {
          method: 'GET',
          headers: {
            Authorization: `Bearer ${token}`,
            'Content-Type': 'application/json',
          },
        });
        if (!response.ok) {
          throw new Error('Failed to fetch bet and win counts');
        }
        const data = await response.json();
        setStats({
          betCount: data.betCount || 0,
          winCount: data.winCount || 0
        });
      } catch (err) {
        console.error('Error fetching bet and win counts:', err);
        setError('Unable to load statistics');
        setStats({ betCount: 0, winCount: 0 });
      } finally {
        setLoading(false);
      }
    };

    fetchBetWinCount();
  }, [show, uid]);

  if (!show) return null;

  return (
    <StyledModal show={show} onClick={onClose}>
      <ModalContent onClick={e => e.stopPropagation()}>
        <ModalHeader>
          <div className="title">User Profile</div>
          <button className="close-button" onClick={onClose}>
            <X size={24} />
          </button>
        </ModalHeader>

        <ModalBody>
          <ProfileSection>
            <img
              src={userData?.avatar || "//img2.distributedresourcestorage.com/avatar/57268122/s?t=0"}
              alt="Profile"
            />
            <h3>{userData?.name || "John Doe"}</h3>
            <div className="level-indicator">
              <Heart size={16} />
              <span>Level {userData?.level || 5}</span>
            </div>
          </ProfileSection>

          <StatisticsSection>
            <h4>Statistics</h4>
            {loading && <p>Loading statistics...</p>}
            {error && <p className="error">{error}</p>}
            {!loading && !error && (
              <StatsGrid>
                <StatBox>
                  <div className="stat-label">Total Wins</div>
                  <div className="stat-value">{stats.winCount ?? userData?.totalWins ?? "1,234"}</div>
                </StatBox>
                <StatBox>
                  <div className="stat-label">Total Bets</div>
                  <div className="stat-value">{stats.betCount ?? userData?.totalBets ?? "2,567"}</div>
                </StatBox>
                <StatBox>
                  <div className="stat-label">Total Wagered</div>
                  <div className="stat-value">${userData?.totalWagered ?? "15,890"}</div>
                </StatBox>
              </StatsGrid>
            )}
          </StatisticsSection>
        </ModalBody>
      </ModalContent>
    </StyledModal>
  );
};

export default UserInfoModal;