import React, { useState, useEffect } from 'react';
import styled from 'styled-components';
import { useBetting } from '../../../../../context/BettingContext';
import { __, wait, decode, encode, forceSatoshiFormat, sendNotfication } from "../../../../../Helper";
import storage from "../../../../../Storage";
import C from "../../../../../Constant";
import socket from "../../../../../Socket";
const Container = styled.div`
  width: 100%;
  background: #000000;
  margin-left: 3px;
`;

const Header = styled.div`
  background: rgb(50, 55, 56);
  color: rgb(240, 156, 6);
  padding: 13.25px 8px;
  display: flex;
  justify-content: space-between;
  align-items: center;
  font-size: 13px;
  font-weight: bold;
  border-bottom: 1px solid rgb(240, 156, 6);
`;

const SubHeader = styled.div`
  display: grid;
  grid-template-columns: 2fr 1fr 1fr 1fr;
  background: rgb(50, 55, 56);
  padding: 6px 8px;
  border-bottom: 1px solid rgb(240, 156, 6);
  font-weight: 500;
  color: #ffffff;
  font-size: 11px;
`;

const MatchTitle = styled.div`
  padding: 6px 8px;
  background: rgb(50, 55, 56);
  display: flex;
  align-items: center;
  gap: 6px;
  color: #ffffff;
  font-size: 11px;
`;

const Dot = styled.span`
  width: 8px;
  height: 8px;
  background: rgb(240, 156, 6);
  border-radius: 50%;
  display: inline-block;
`;

const BetRowContainer = styled.div`
  display: grid;
  grid-template-columns: 2fr 1fr 1fr 1fr;
  padding: 6px 8px;
  background: #000000;
  align-items: center;
  gap: 6px;
  border-bottom: 1px solid rgb(50, 55, 56);
`;


const BetRowContainerBlue = styled.div`
  display: grid;
  grid-template-columns: 2fr 1fr 1fr 1fr;
  padding: 6px 8px;
  background: #000000;
  align-items: center;
  gap: 6px;
  border-bottom: 1px solid rgb(50, 55, 56);
`;

const TeamSection = styled.div`
  display: flex;
  align-items: center;
  gap: 4px;
`;

const CloseIcon = styled.span`
  color: rgb(240, 156, 6);
  margin-right: 3px;
  cursor: pointer;
  font-size: 12px;
`;

const TeamName = styled.span`
  margin-right: 3px;
  font-size: 10px !important;
  color: #ffffff;
`;

const MatchOdds = styled.span`
  color: #ffffff;
  font-size: 0.8em;
`;

const OddsInputContainer = styled.div`
  position: relative;
  width: 100%;
`;

const StyledInput = styled.input`
  width: 100%;
  padding: 4px;
  border: 1px solid rgb(50, 55, 56);
  border-radius: 3px;
  text-align: right;
  font-size: 0.8em;
  outline: none;
  color: #ffffff;
  background: rgb(50, 55, 56);
  &:focus {
    border-color: rgb(240, 156, 6);
  }
`;

const ArrowsContainer = styled.div`
  position: absolute;
  right: 5px;
  top: 50%;
  transform: translateY(-50%);
  display: flex;
  flex-direction: column;
  gap: 2px;
`;

const Arrow = styled.span`
  cursor: pointer;
  font-size: 8px;
  color: rgb(240, 156, 6);
  line-height: 1;
  &:hover {
    color: #ffffff;
  }
`;

const LiabilityValue = styled.div`
  text-align: right;
  padding: 4px 6px;
  background: rgb(50, 55, 56);
  border: 1px solid rgb(240, 156, 6);
  border-radius: 3px;
  min-width: 30px;
  color: rgb(240, 156, 6);
  font-weight: 500;
  height: 24px;
  display: flex;
  align-items: center;
  justify-content: flex-end;
  font-size: 0.8em;
`;

const QuickStakeButtons = styled.div`
  display: grid;
  grid-template-columns: repeat(6, 1fr);
  gap: 4px;
  padding: 6px 8px;
  background: #000000;
`;
const QuickStakeButtonsBlue = styled.div`
  display: grid;
  grid-template-columns: repeat(6, 1fr);
  gap: 4px;
  padding: 6px 8px;
  background: #000000;
`;


const StakeButton = styled.div`
  padding: 4px;
  background: rgb(50, 55, 56);
  display: flex;
  align-items: center;
  justify-content: center;
  border: 1px solid rgb(240, 156, 6);
  color: #ffffff;
  border-radius: 3px;
  cursor: pointer;
  font-size: 0.8em;
  &:hover {
    background: rgb(60, 65, 66);
  }
`;

const Footer = styled.div`
  padding: 6px 8px;
  background: #000000;
`;
const FooterBlue = styled.div`
  padding: 6px 8px;
  background: #000000
`;


const ActionButtons = styled.div`
  display: grid;
  grid-template-columns: 1fr 1fr;
  gap: 6px;
  padding: 6px;
`;

const Button = styled.button`
  padding: 6px 8px;
  border-radius: 3px;
  border: none;
  cursor: pointer;
  font-weight: bold;
  font-size: 0.8em;
  
  ${props => props.primary ? `
    background: rgb(240, 156, 6);
    color: #000000;
    &:hover {
      background: rgb(220, 136, 6);
    }
  ` : `
    background: rgb(50, 55, 56);
    border: 1px solid rgb(240, 156, 6);
    color: #ffffff;
    &:hover {
      background: rgb(60, 65, 66);
    }
  `}
`;

const Button1 = styled.div`
  padding: 10px;
  border-radius: 4px;
  border: none;
  cursor: pointer;
  font-weight: bold;
  display: flex;
  color: #ffffff;
  align-items: center;
  justify-content: center;
  font-size: 0.85em;
  
  ${props => props.primary ? `
    background: rgb(240, 156, 6);
    color: #000000;
    &:hover {
      background: rgb(220, 136, 6);
    }
  ` : `
    background: rgb(50, 55, 56);
    border: 1px solid rgb(240, 156, 6);
    color: #ffffff;
    &:hover {
      background: rgb(60, 65, 66);
    }
  `}
`;

const BetConfirm = styled.div`
  padding: 6px 8px;
  display: flex;
  align-items: center;
  gap: 6px;
  color: #ffffff;
  font-size: 11px;
  
  span {
    color: #ffffff;
  }
`;

const MinBetText = styled.div`
  text-align: right;
  padding: 3px;
  color: #ffffff;
  font-size: 10px;
`;

const LiabilityText = styled.div`
  text-align: right;
  padding: 3px;
  display: flex;
  justify-content: flex-end;
  gap: 6px;
  color: rgb(240, 156, 6);
  font-size: 11px;
  font-weight: bold;
`;

const BetSlip = () => {
  const [betsState, setBetsState] = useState([]);
  const [isConfirmed, setIsConfirmed] = useState(false);
  const { selectedBets, removeBet, placeBets, updateBetStake, updateBetOdds } = useBetting();
  const [totalLiability, setTotalLiability] = useState(0);
 const [selectedCoin, setSelectedCoin] = useState(storage.getKey('coin') || "INR");

  useEffect(() => {
    const token = storage.getKey('token');

    if (token) {
      socket.emit(C.CREDIT, encode({ token, coin: selectedCoin }));
    }

    const handleCreditData = (data) => handleCredit(decode(data));
    const handleCreditDataUpdate = (data) => handleCreditUpdate(decode(data));

    socket.on(C.CREDIT, handleCreditData);
    socket.on(C.UPDATE_CREDIT, handleCreditDataUpdate);

    return () => {
      socket.off(C.CREDIT, handleCreditData);
      socket.off(C.UPDATE_CREDIT, handleCreditUpdate);
    };
  }, []);
 const handleCredit = (data) => {
    const { credit } = data;
    const currentCoin = __.lowerCase("USDT");
    const currentCredit = forceSatoshiFormat(credit[currentCoin]);
    console.log('current credit ' +currentCredit);
    storage.setKey('credit', currentCredit);
  };
const handleCreditUpdate = (data) => {
    console.log('update credit' + data);
    const { value, coin: updatedCoin } = data;
    const formattedValue = forceSatoshiFormat(value);
    if (__.lowerCase(updatedCoin) === __.lowerCase(selectedCoin)) {
      storage.setKey('credit', formattedValue);
    }
  };

  useEffect(() => {
    if (selectedBets?.length > 0) {
      setBetsState(selectedBets);
    } else {
      setBetsState([]);
    }
  }, [selectedBets]);

  useEffect(() => {
    const total = selectedBets.reduce((sum, bet) => {
      if (!bet.stake || !bet.odds) return sum;
      const stake = parseFloat(bet.stake) || 0;
      const odds = parseFloat(bet.odds) || 0;
      const liability = bet.type === 'lay' ? (stake * (odds - 1)) : stake;
      return sum + liability;
    }, 0);
    setTotalLiability(total.toFixed(2));
  }, [selectedBets]);

  const handleStakeChange = (value, betId) => {
    const validValue = value.replace(/[^0-9.]/g, '');
    updateBetStake(betId, validValue);

    // Update local state to reflect changes immediately
    setBetsState(prevState =>
      prevState.map(bet => {
        if (bet.id === betId) {
          const stake = parseFloat(validValue) || 0;
          const odds = parseFloat(bet.odds) || 0;
          const profit = (stake * (odds - 1)).toFixed(2);
          return {
            ...bet,
            stake: validValue,
            profit: profit
          };
        }
        return bet;
      })
    );
  };
  const handleQuickStake = (value, betId) => {
    handleStakeChange(value.toString(), betId);
  };

  const calculateProfit = (bet) => {
    if (!bet.stake || !bet.odds) return '0.00';
    const stake = parseFloat(bet.stake) || 0;
    const odds = parseFloat(bet.odds) || 0;
    return (stake * (odds - 1)).toFixed(2);
  };


 
 
  const handleOddsChange = (newOdds, betId) => {
    setBetsState(prevState =>
      prevState.map(bet => {
        if (bet.id === betId) {
          const stake = parseFloat(bet.stake) || 0;
          const odds = parseFloat(newOdds) || 0;
          const profit = (stake * (odds - 1)).toFixed(2);
          return {
            ...bet,
            odds: newOdds,
            profit: profit
          };
        }
        return bet;
      })
    );

    // Update context to reflect odds change
    updateBetOdds(betId, newOdds);
  };

  const handleCancelAll = () => {
    selectedBets.forEach(bet => removeBet(bet.id));
    setBetsState([]);
    setIsConfirmed(false);
  };

  const handlePlaceBets = async () => {
    if (!isConfirmed) {
      // alert('Please confirm your bets before placing them.');
      sendNotfication('Please confirm your bets before placing them.', 'error', 'bottom-left');
      return;
    }

    // Validate bets
    const validBets = betsState.filter(bet =>
      bet.stake &&
      parseFloat(bet.stake) >= 4 &&
      bet.odds &&
      parseFloat(bet.odds) > 1
    );

    if (validBets.length === 0) {
      // alert('Please enter valid stakes and odds for your bets.');
      sendNotfication('Please enter valid stakes and odds for your bets.', 'error', 'bottom-left');
      return;
    }

    // Place bets via context (server-side)
    const result = await placeBets(validBets);

    // Log result
    console.log('PlaceBets result:', result);

    if (result?.confirmed?.length > 0) {
      // alert('Bets placed successfully!');
      sendNotfication('Bets placed successfully!', 'success', 'bottom-left')
      // Clear only confirmed from slip handled by context; local UI already synced
    } else {
      const firstError = result?.errors?.[0]?.message || 'Bet placement failed';
      // alert(firstError);
       sendNotfication(firstError, 'error', 'bottom-left')
      return;
    }

    // Clear bet slip (any remaining unconfirmed bets are still visible)
    if ((selectedBets?.length || 0) === 0) {
      handleCancelAll();
    }
  };

  return (
    <Container>
      <Header>
        <span>Bet Slip ({selectedBets?.length || 0})</span>
      </Header>

      {selectedBets?.length === 0 ? (
        <div style={{ padding: '8px', textAlign: 'center', color: '#ffffff', borderBottomWidth: '1px', borderBottomColor: 'rgb(50, 55, 56)', borderBottomStyle: 'solid' }}>
          No Bets
        </div>
      ) : (
        betsState.map((bet) => (
          <div key={bet.id}>
            <SubHeader>
              {bet?.type === "lay" ?
                <div>Lay (Bet Against)</div> :
                <div>Back (Bet For)</div>
              }
              <div>Odds</div>
              <div>Stake</div>
              {bet?.type === "lay" ?
                <div>Liability</div> :
                <div>Profit</div>
              }
            </SubHeader>

            <MatchTitle>
              <Dot />
              {bet?.matchTitle}
            </MatchTitle>

            {bet?.type === "lay" ? (
              <>
                <BetRowContainer>
                  <TeamSection>
                    <CloseIcon onClick={() => removeBet(bet.id)}>✕</CloseIcon>
                    <TeamName>{bet.selectionName}</TeamName>
                    <MatchOdds>{bet.category}</MatchOdds>
                  </TeamSection>
                  <OddsInputContainer>
                    <StyledInput
                      type="text"
                      value={bet.odds}
                      onChange={(e) => handleOddsChange(e.target.value, bet.id)}
                      style={{ paddingRight: '20px' }}
                    />
                    <ArrowsContainer>
                      <Arrow onClick={() => handleOddsChange((parseFloat(bet.odds) + 0.01).toFixed(2), bet.id)}>▲</Arrow>
                      <Arrow onClick={() => handleOddsChange((parseFloat(bet.odds) - 0.01).toFixed(2), bet.id)}>▼</Arrow>
                    </ArrowsContainer>
                  </OddsInputContainer>
                  <StyledInput
                    type="text"
                    value={bet.stake}
                    onChange={(e) => handleStakeChange(e.target.value, bet.id)}
                    placeholder="0"
                  />
                  <LiabilityValue>
                    {bet.liability}
                  </LiabilityValue>
                </BetRowContainer>
                <QuickStakeButtons>
                  {[10, 50, 100, 200, 500, 1000].map(value => (
                    <StakeButton key={value} onClick={() => handleQuickStake(value, bet.id)}>
                      {value}
                    </StakeButton>
                  ))}
                </QuickStakeButtons>
                <Footer>
                  <MinBetText>Min Bet: 4</MinBetText>
                </Footer>
              </>
            ) : (
              <>
                <BetRowContainerBlue>
                  <TeamSection>
                    <CloseIcon onClick={() => removeBet(bet.id)}>✕</CloseIcon>
                    <TeamName>{bet.selectionName}</TeamName>
                    <MatchOdds>{bet.category}</MatchOdds>
                  </TeamSection>
                  <OddsInputContainer>
                    <StyledInput
                      type="text"
                      value={bet.odds}
                      onChange={(e) => handleOddsChange(e.target.value, bet.id)}
                      style={{ paddingRight: '20px' }}
                    />
                    <ArrowsContainer>
                      <Arrow onClick={() => handleOddsChange((parseFloat(bet.odds) + 0.01).toFixed(2), bet.id)}>▲</Arrow>
                      <Arrow onClick={() => handleOddsChange((parseFloat(bet.odds) - 0.01).toFixed(2), bet.id)}>▼</Arrow>
                    </ArrowsContainer>
                  </OddsInputContainer>
                  <StyledInput
                    type="text"
                    value={bet.stake}
                    onChange={(e) => handleStakeChange(e.target.value, bet.id)}
                    placeholder="0"
                  />
                  <LiabilityValue>
                    {bet.liability}
                  </LiabilityValue>
                </BetRowContainerBlue>
                <QuickStakeButtonsBlue>
                  {[10, 50, 100, 200, 500, 1000].map(value => (
                    <StakeButton key={value} onClick={() => handleQuickStake(value, bet.id)}>
                      {value}
                    </StakeButton>
                  ))}
                </QuickStakeButtonsBlue>
                <FooterBlue>
                  <MinBetText>Min Bet: 4</MinBetText>
                </FooterBlue>
              </>
            )}
          </div>
        ))
      )}

      {selectedBets?.length > 0 && (
        <>
          <LiabilityText>Total Liability: {totalLiability}</LiabilityText>
          <ActionButtons>
            <Button1 onClick={handleCancelAll}>Cancel All</Button1>
            <Button primary onClick={handlePlaceBets}>Place Bets</Button>
          </ActionButtons>

          <BetConfirm>
            <input
              type="checkbox"
              checked={isConfirmed}
              onChange={(e) => setIsConfirmed(e.target.checked)}
            />
            <span>Please confirm your bets.</span>
          </BetConfirm>
        </>
      )}
    </Container>
  );
};


export default BetSlip;

