import React, { useState, useEffect } from 'react';
import styled from 'styled-components';
import { useBetting } from '../../../../../context/BettingContext';
import { useHistory } from 'react-router-dom';

// Styled Components (reuse exactly as in your Today component)
const Container = styled.div`
  width: 100%;
  background: #000000;
  font-family: 'Montserrat', sans-serif !important;
  display: flex;
  flex-direction: column;
  height: 100vh;

  /* Force all descendants to use Montserrat */
  *, *::before, *::after {
    font-family: 'Montserrat', sans-serif !important;
  }
`;

const Header = styled.div`
  background: rgb(50, 55, 56);
  color: rgb(240, 156, 6);
  padding: 10px;
  font-size: 18px;
  font-weight: bold;
  font-family: 'Montserrat', sans-serif !important;
`;

const Content = styled.div`
  background: #000000;
  flex: 1;
  overflow-y: auto;
  -webkit-overflow-scrolling: touch;

  &::-webkit-scrollbar { width: 8px; }
  &::-webkit-scrollbar-track { background: rgb(50, 55, 56); }
  &::-webkit-scrollbar-thumb {
    background: rgb(240, 156, 6);
    border-radius: 4px;
  }
  &::-webkit-scrollbar-thumb:hover {
    background: rgb(220, 136, 6);
  }
`;

const Match = styled.div`
  display: flex;
  justify-content: space-between;
  align-items: center;
  padding: 4px 10px;
  border-bottom: 1px solid rgb(50, 55, 56);
  background: #000000;
  cursor: pointer;
`;

const MatchInfo = styled.div` flex: 1; `;

const TitleContainer = styled.div`
  display: flex;
  align-items: center;
`;

const LiveDot = styled.span`
  display: inline-block;
  width: 8px; height: 8px;
  margin-right: 8px;
  background-color: #2ecc71;
  border-radius: 50%;
  animation: blink 1s ease-in-out infinite;
  @keyframes blink {
    0%,100% { opacity: 0; }
    50% { opacity: 1; }
  }
`;

const NOLiveDot = styled.span`
  display: inline-block;
  width: 8px; height: 8px;
  margin-right: 8px;
  background-color: #cccccc;
  border-radius: 50%;
`;

const MatchTitle = styled.div`
  color: #ffffff;
  font-weight: bold;
  font-size: 11px;
  font-family: 'Montserrat', sans-serif !important;
`;

const Status = styled.span`
  color: rgb(240, 156, 6);
  font-size: 12px;
  margin-left: 14px;
  font-weight: bold;
  font-family: 'Montserrat', sans-serif !important;
`;

const NoStatus = styled.span`
  color: #cccccc;
  font-size: 10px;
  margin-left: 14px;
  font-family: 'Montserrat', sans-serif !important;
`;

const PTE = styled.div`
  color: #ffffff;
  font-size: 12px;
  margin-right: 10px;
  font-family: 'Montserrat', sans-serif !important;
`;

const IconGroup = styled.div`
  display: flex;
  gap: 4px;
  margin-left: 8px;
`;

const SportsScheduleTomorrow = () => {
  const [matches, setMatches] = useState([]);
  const { addBet } = useBetting();      // if unused you can remove
  const history = useHistory();

  useEffect(() => {
    const fetchMatches = async () => {
      try {
        const token = localStorage.getItem('token');
        const res = await fetch(
          'https://api.boss707.com/sports/matches-by-date/tomorrow',
          {
            headers: {
              'Authorization': `Bearer ${token}`,
              'Content-Type': 'application/json'
            }
          }
        );
        const data = await res.json();
        const allMarkets = data.flatMap(sport => sport.markets || []);

        const mapped = allMarkets.map(m => {
          const utcDate = new Date(m.marketStartTime);
          const istTime = new Intl.DateTimeFormat('en-IN', {
            hour: '2-digit',
            minute: '2-digit',
            hour12: true
          }).format(utcDate);

          const team1 = m.runners[0]?.runnerName || '';
          const team2 = m.runners[1]?.runnerName || '';
          const pteSum = m.runners.reduce((sum, r) => sum + (r.totalMatched || 0), 0);
          const pteLabel = `PTE${(pteSum / 1e6).toFixed(1)}M`;

          return {
            id: m.marketId,
            team1,
            team2,
            status: m.inplay ? 'In-Play' : istTime,
            isLive: m.inplay,
            pte: pteLabel,
            isSuspended: m.stopBet
          };
        });

        setMatches(mapped);
      } catch (err) {
        console.error('Failed to load tomorrow’s matches', err);
      }
    };

    fetchMatches();
  }, []);

  const handleMatchClick = match => {
    history.push({
      pathname: `/sports/markets/${match.id}`,
      state: { matchData: match }
    });
  };

  return (
    <Container>
      <Header>Tomorrow</Header>
      <Content>
        {matches.map((match, idx) => (
          <Match key={idx} onClick={() => handleMatchClick(match)}>
            <MatchInfo>
              <TitleContainer>
                {match.isLive ? <LiveDot /> : <NOLiveDot />}
                <MatchTitle>{`${match.team1} v ${match.team2}`}</MatchTitle>
              </TitleContainer>
              <div style={{ display: 'flex', alignItems: 'center' }}>
                {match.isLive ? (
                  <Status>In-Play</Status>
                ) : (
                  <NoStatus>{match.status}</NoStatus>
                )}
                <PTE>{match.pte}</PTE>
                <IconGroup>
                  <img
                    src="/assets/images/sports/cricket.png"
                    alt="sport"
                    style={{ width: '20px', height: '20px', objectFit: 'contain' }}
                  />
                  <img
                    src="/assets/images/sports/livestream.png"
                    alt="live"
                    style={{ width: '20px', height: '20px', objectFit: 'contain' }}
                  />
                  <img
                    src="/assets/images/sports/watch.png"
                    alt="watch"
                    style={{ width: '20px', height: '20px', objectFit: 'contain' }}
                  />
                </IconGroup>
              </div>
            </MatchInfo>
          </Match>
        ))}
      </Content>
    </Container>
  );
};

export default SportsScheduleTomorrow;
