import React, { useCallback, useEffect, useState } from 'react';
import styled, { keyframes } from 'styled-components';
import { ChevronRight, ChevronDown, Search, Info, Copy, Upload, CheckCircle } from "lucide-react";
import storage from "../../../../Storage";
import C from "../../../../Constant";
import coins from "../../../coins";
import socket from "../../../../Socket";
import { Col, Row, Dropdown } from "react-bootstrap";
import {
  __, wait, decode, encode, forceSatoshiFormat, Event,
  isMobile,
  sendNotfication,
  getUID,
} from '../../../../Helper';
import axios from 'axios';
import { QRCode } from 'react-qrcode-logo';
import Storage from '../../../../Storage';
import { Info as InfoIcon } from 'lucide-react';


const Container = styled.div`
  padding: 8px;
  color: #ffffff;
  background: #1B2326;
  @media (max-width: 768px) {
    padding: 0px;
  }
`;

const DropdownContainer = styled.div`
  position: relative;
  margin-bottom: 8px;
  flex: 1;
`;

const DropdownButton = styled.button`
  display: flex;
  align-items: center;
  justify-content: space-between;
  width: 100%;
  padding: 6px 10px;
  background: #1B2326;
  border: 1px solid rgba(255, 255, 255, 0.05);
  border-radius: 8px;
  color: #ffffff;
  cursor: pointer;
  font-size: 16px;
`;

const DropdownPanel = styled.div`
  position: absolute;
  top: 100%;
  left: 0;
  right: 0;
  background: #1B2326;
  border: 1px solid rgba(255, 255, 255, 0.05);
  border-radius: 8px;
  margin-top: 4px;
  max-height: 400px;
  overflow-y: auto;
  z-index: 1000;
  padding: 10px 12px;
  @media (max-width: 768px) {
    max-height: 300px;
    width: 100%;
    z-index: 1001;
  }
`;

const SearchContainer = styled.div`
  position: relative;
  padding: -6px;
  border-radius: 8px;
  border: 1px solid rgba(255, 255, 255, 0.05);
  margin-top: 12px;
  @media (max-width: 768px) {
    margin-top: 8px;
    margin-bottom: 8px;
  }
`;

const SearchInput = styled.input`
  width: 100%;
  padding: 6px 12px 6px 40px;
  background: transparent;
  border: none;
  border-radius: 8px;
  color: #ffffff;
  font-size: 16px;
  &::placeholder {
    color: #fff;
    opacity: 0.6;
  }
  @media (max-width: 768px) {
    padding: 6px 8px 6px 36px;
    font-size: 14px;
  }
`;

const Input = styled.input`
  flex: 1;
  padding: 6px 10px;
  background: #192225;
  border: 1px solid rgba(255, 255, 255, 0.05);
  border-radius: 8px;
  color: #ffffff;
  font-size: 14px;
  &::placeholder {
    color: #fff;
    opacity: 0.6;
  }
`;

const DepositButton = styled.div`
  margin-top: auto;
  flex: 1;
  padding: 7px 9px;
  display: flex;
  justify-content: center;
  align-items: center;
  background: #F79C03; /* Keeping original orange color */
  color: black !important;
  border: none;
  border-radius: 0.5rem;
  font-weight: 600;
  cursor: pointer;
  &:hover {
    opacity: 0.9;
  }
`;

const SearchIcon = styled.div`
  position: absolute;
  left: 14px;
  top: 49%;
  transform: translateY(-50%);
  color: #fff;
`;

const NoteContainer = styled.div`
  background: rgba(0, 0, 0, 0.2);
  border-radius: 8px;
  padding: 12px 16px;
  display: flex;
  align-items: center;
  gap: 12px;
  margin: 8px auto 16px;
  border-left: 3px solid #FF9D02;
`;

const NoteText = styled.span`
  color: #ffffff;
  font-size: 14px;
  font-weight: 400;
  line-height: 1.5;
`;

const NoteIcon = styled(Info)`
  color: #FF9D02;
  flex-shrink: 0;
`;

const CoinOption = styled.div`
  display: flex;
  align-items: center;
  padding: 14px;
  cursor: pointer;
  border-radius: 12px;
  &:hover {
    background: rgba(0, 0, 0, 0.25);
    border-color: rgba(255, 255, 255, 0.1);
  }
  ${props => props.isSelected && `
    background: rgba(0, 0, 0, 0.25);
    border-color: rgba(255, 255, 255, 0.1);
  `}
  @media (max-width: 768px) {
    padding: 10px;
    margin-bottom: 4px;
  }
`;

const CoinImage = styled.img`
  width: 24px;
  height: 24px;
  border-radius: 50%;
  margin-right: 12px;
  @media (max-width: 768px) {
    width: 20px;
    height: 20px;
    margin-right: 8px;
  }
`;

const CoinName = styled.div`
  font-size: 16px;
  font-weight: bold;
  @media (max-width: 768px) {
    font-size: 14px;
  }
`;

const TokenListContainer = styled.div`
  display: flex;
  align-items: center;
  gap: 8px;
  margin-top: 16px;
  @media (max-width: 768px) {
    gap: 4px;
    margin-top: 4px;
  }
`;

const TokenButton = styled.button`
  display: flex;
  align-items: center;
  gap: 8px;
  padding: 8px 16px;
  background: #1B2326;
  border-radius: 100px;
  border: none;
  cursor: pointer;
  color: #ffffff;
  transition: background 0.2s ease;
  &:hover {
    background: #1B2326;
  }
  @media (max-width: 768px) {
    gap: 4px;
    padding: 4px 8px;
  }
`;

const TokenIcon = styled.img`
  width: 24px;
  height: 24px;
  border-radius: 50%;
  @media (max-width: 768px) {
    width: 18px;
    height: 18px;
  }
`;

const TokenName = styled.div`
  font-size: 14px;
  font-weight: 500;
  @media (max-width: 768px) {
    font-size: 12px;
  }
`;

const QRCodeContainer = styled.div`
  width: 100%;
  border-radius: 4px;
  overflow: hidden;
  display: flex;
  flex-direction: column;
  align-items: center;
  justify-content: center;
  img {
    width: 205px;
    height: 205px;
    object-fit: contain;
  }

  @media (max-width: 768px) {
    img {
    width: 120px;
    height: 120px;
    object-fit: contain;
  }
  }
`;

const ContentContainer = styled.div`
  width: 100%;
  display: flex;
  flex-direction: column;
  gap: 10px;
  @media (max-width: 768px) {
    width: 61%;
  }
`;

const Label = styled.div`
  color: #9ba1a6;
  font-size: 14px;
`;

const AddressContainer = styled.div`
  background: #192323;
  padding: 12px;
  border-radius: 6px;
  width: 100%;
  color: #fff;
  font-family: monospace;
  font-size: 14px;
  display: flex;
  justify-content: space-between;
  align-items: center;
  gap: 10px;
  span {
    overflow: hidden;
    text-overflow: ellipsis;
    white-space: nowrap;
    min-width: 0;
  }
`;

const CopyButton = styled.div`
  background: rgba(0, 0, 0, 0.35);
  color: #fff;
  border: none;
  // flex: 1;
  width:100%;
  padding: 8px 16px;
  border-radius: 4px;
  cursor: pointer;
  display: flex;
  text-align: center;
  align-items: center;
  gap: 8px;
  font-size: 14px;
  transition: background-color 0.2s;
  white-space: nowrap;
  width: fit-content;
`;

const ErrorText = styled.div`
  color: #ff4d4f;
  font-size: 12px;
  margin-top: 0px;
`;

const TransactionForm = styled.div`
  display: flex;
  flex-direction: column;
  gap: 12px;
  margin-top: 16px;
  width: 100%;
`;

const FileUploadContainer = styled.div`
  border: 2px dashed #41ED84;
  border-radius: 8px;
  padding: 16px;
  text-align: center;
  cursor: pointer;
  margin-top: 8px;
  &:hover {
    border-color: #20EE89;
  }
`;

const UploadText = styled.div`
  color: #ffffff;
  margin-top: 8px;
  font-size: 14px;
`;

const spin = keyframes`
  0% { transform: rotate(0deg); }
  100% { transform: rotate(360deg); }
`;

const pulse = keyframes`
  0%, 100% { opacity: 1; }
  50% { opacity: 0.5; }
`;

const SpinnerWrapper = styled.div`
  display: flex;
  align-items: center;
  justify-content: center;
  gap: 2;
`;

const Spinner = styled.div`
  width: 2rem;
  height: 2rem;
  border: 3px solid #e5e7eb;
  border-top-color: #ab1ac6;
  border-radius: 50%;
  animation: ${spin} 0.8s linear infinite;
`;

const LoadingText = styled.span`
  color: #ab1ac6;
  font-size: 1rem;
  margin-left: 0.75rem;
  animation: ${pulse} 1.5s ease-in-out infinite;
`;

const PaymentMethodContainer = styled.div`
  display: flex;
  flex-direction: column;
  gap: 16px;
  margin-top: 16px;
  width: 100%;
`;

const MethodSelector = styled.div`
  display: flex;
  gap: 12px;
  width: 100%;
`;

const MethodButton = styled.button`
  flex: 1;
  padding: 12px;
  border-radius: 8px;
  border: 2px solid ${props => props.selected ? '#F79C03' : '#F79C03'};
  background: ${props => props.selected ? '#F79C03' : 'transparent'};
  color: ${props => props.selected ? '#000000' : '#ffffff'};
  font-weight: 500;
  cursor: pointer;
  transition: all 0.2s;
  &:hover {
    background: ${props => props.selected ? '#F79C03' : 'rgba(247, 156, 3, 0.1)'};
  }
`;

const WarningContainer = styled.div`
  background: rgba(0, 0, 0, 0.2);
  border-radius: 8px;
  padding: 12px 16px;
  display: flex;
  align-items: center;
  gap: 12px;
  margin: 0 auto;
  margin-bottom: 16px;
`;

const WarningText = styled.span`
  color: #ffff;
  font-size: 14px;
  font-weight: 500;
`;

const tokens = [
  { id: 'eth', name: 'ETH', icon: '/assets/images/ETH.png' },
  { id: 'btc', name: 'BTC', icon: '/assets/images/BTC.png' },
  { id: 'usdt', name: 'USDT', icon: '/assets/images/USDT.png' },
  { id: 'doge', name: 'DOGE', icon: '/assets/images/DOGE.png' },
];

// Bonus Card Styled Components (copied from ModalContent.js)
const BonusCardsWrapper = styled.div`
  display: flex;
  flex-direction: row;
  gap: 10px;
  justify-content: left;
  margin: 18px 0 12px 0;
  @media (max-width: 768px) {
    flex-direction: row;
    gap: 6px;
    align-items: stretch;
  }
`;

const BonusCardContainer = styled.div`
  display: flex;
  // flex-direction: row;
  align-items: center;
  justify-content: center;
  background: #1B2326;
  gap: 20px;
  border: 1px solid transparent;
  border-radius: 12px;
  box-shadow: 0 8px 32px rgba(0,0,0,0.18);
  // padding: 14px 10px 14px 10px;
  max-width: 200px;
  min-width: 120px;
  min-height: 50px;
  width: 100%;
  position: relative;
  cursor: pointer;
  transition: border 0.2s;
  &.selected {
    border: 1px solid #F79C03;
  }
  @media (max-width: 768px) {
    max-width: 48vw;
    min-width: 0;
    padding: 12px;
    min-height: 70px;
  }
`;
const BonusSVGWrapper = styled.div`
  display: flex;
  align-items: center;
  justify-content: center;
  // margin-bottom: 2px;
  // margin-top: 2px;
`;
const InfoIconButton = styled.div
`
  // position: absolute;
  // top: 16px;
  // right: 16px;
  // background: none;
  // border: none;
  // cursor: pointer;
  // color: #fff;
  // padding: 4px;
  // z-index: 2;
`;
const BonusTypeToggle = styled.div`
  display: flex;
  gap: 16px;
  margin-top: 2px;
`;
const BonusToggleButton = styled.button`
  padding: 4px 24px;
  border-radius: 8px;
  border: none;
  font-weight: 600;
  font-size: 14px;
  background: ${({ selected }) => (selected ? '#F79C03' : '#23272f')};
  color: ${({ selected }) => (selected ? '#23272f' : '#fff')};
  cursor: ${({ disabled }) => (disabled ? 'not-allowed' : 'pointer')};
  opacity: ${({ disabled }) => (disabled ? 0.7 : 1)};
  transition: background 0.2s, color 0.2s;
`;
const BonusInfoModalOverlay = styled.div`
  position: fixed;
  top: 0;
  left: 0;
  right: 0;
  bottom: 0;
  background: rgba(0,0,0,0.7);
  z-index: 9999;
  display: flex;
  align-items: center;
  justify-content: center;
`;
const BonusInfoModalContent = styled.div`
  background: #23272f;
  border-radius: 16px;
  padding: 32px 24px 24px 24px;
  max-width: 420px;
  width: 90vw;
  color: #fff;
  position: relative;
`;
const BonusInfoModalClose = styled.button`
  position: absolute;
  top: 16px;
  right: 16px;
  background: none;
  border: none;
  color: #fff;
  font-size: 24px;
  cursor: pointer;
`;
const BonusCardTitle = styled.div`
  text-align: center;
  margin-bottom: 8px;
  font-weight: 700;
  font-size: 16px;
  letter-spacing: 0.5px;
`;
const BonusCardInfo = styled.div`
  text-align: center;
  margin-bottom: 6px;
  font-size: 13px;
`;
const BonusCardHighlight = styled.span`
  color: #F79C03;
  font-weight: 600;
  font-size: 14px;
`;

const BonusCardSectionTitle = styled.div`
  text-align: center;
  font-size: 16px;
  font-weight: 700;
  margin-bottom: 2px;
  letter-spacing: 1px;
  color: #F79C03;
`;

const ImagePlaceholder = styled.img`
  width: 120px;
  height: 120px;
  object-fit: contain;
`;

const rotate = keyframes`
  0% { transform: rotate(0deg); }
  100% { transform: rotate(360deg); }
`;

const RotatingImage = styled(ImagePlaceholder)`
  animation: ${rotate} 6s linear infinite;
`;

const SmallImagePlaceholder = styled.img`
  width: 25px;
  height: 25px;
  object-fit: contain;
  @media (max-width: 768px) {
    width: 20px;
    height: 20px;
  }
`;
const SmallRotatingImage = styled(SmallImagePlaceholder)`
  animation: ${rotate} 6s linear infinite;
`;

const SkylaDeposit = () => {
  const [isOpen, setIsOpen] = useState(false);
  const [isOpenNetwork, setIsOpenNetwork] = useState(false);
  const [searchTerm, setSearchTerm] = useState("");
  const [selectedNetwork, setSelectedNetwork] = useState("");
  const [selectedBank, setSelectedBank] = useState("Bank Of Maldives");
  const [error, setError] = useState("");
  const [showTransactionForm, setShowTransactionForm] = useState(false);
  const [transactionId, setTransactionId] = useState("");
  const [screenshot, setScreenshot] = useState(null);
  const [phoneNumber, setPhoneNumber] = useState(""); // Added for automatic payment
  const [selectedCoin, setSelectedCoin] = useState(Storage.getKey('coin') || "INR");
  const [activeCoins, setActiveCoins] = useState(
    storage.getKey('active_coins')?.split(',').map(String) || ['1']
  );
  const [paymentMode, setPaymentMode] = useState('automatic');
  const [state, setState] = useState({
    currentCoin: storage.getKey('coin') || "BTC",
    credits: {},
    list: [],
    content: [],
    wallet: "btc",
    deposit: true,
    slide: false,
    height: 573,
    margin: "mt-1",
    amount: "",
    coinId: "",
    networks: [],
    selectedNetwork: "",
    qrString: "",
    coinName: "",
    logoUrl: "",
    isLoading: false,
    isCopied: false,
    depositAddress: "",
    depositAmount: "",
    showDepositDetails: false,
    iframeUrl: '', // New state for iframe URL
    showIframe: false // New state to toggle iframe visibility
  });
  const [bankDetails, setBankDetails] = useState([]);
  const [selectedBankDetails, setSelectedBankDetails] = useState(null);
  const [copiedStates, setCopiedStates] = useState({
    accountName: false,
    accountNumber: false,
    ifscCode: false,
    upiId: false
  });
  const [ccInitiate, setCCInitiate] = useState(false);

  // Bonus Card State
  const [bonusConfig, setBonusConfig] = useState([]);
  const [selectedBonusType, setSelectedBonusType] = useState('casino');
  const [showBonusInfoModal, setShowBonusInfoModal] = useState(false);
  const [selectedBonusCard, setSelectedBonusCard] = useState(null); // 'casino' or 'sports' or null

  // Add state to control bonus card visibility
  const [showBonusCards, setShowBonusCards] = useState(false);

  const FIAT_CURRENCIES = {
    INR: { payType: 'UPI', automatic: true, manual: true },
    MVR: { payType: 'Bank', automatic: false, manual: true },
    PKR: { payType: 'Bank', automatic: false, manual: true },
    AED: { payType: 'Bank', automatic: false, manual: true },
    BDT: { payType: 'Bank', automatic: false, manual: true }
  };

  const isFiatCurrency = (currency) => Object.keys(FIAT_CURRENCIES).includes(currency);

  useEffect(() => {
    if (isFiatCurrency(selectedCoin)) {
      fetchBankDetails();
    }
  }, [selectedCoin, paymentMode]);

  const fetchBankDetails = async () => {
    try {
      const token = localStorage.getItem('token');
      const response = await fetch(`https://api.boss707.com/bankDetails/${selectedCoin}`, {
        headers: {
          'Authorization': `Bearer ${token}`,
        },
      });
      if (!response.ok) throw new Error('Failed to fetch bank details');
      const data = await response.json();
      setBankDetails(data);
      if (data.length > 0) {
        setSelectedBank(data[0].bank_name);
        setSelectedBankDetails({
          accountName: data[0].account_holder_name,
          accountNumber: data[0].account_number,
          ifscCode: data[0].ifsc_code,
          upiId: data[0].upi_id
        });
      }
    } catch (error) {
      console.error('Error fetching bank details:', error);
      setError('Failed to load bank details. Please try again.');
    }
  };

  const handleFileUpload = (event) => {
    const file = event.target.files[0];
    if (file) {
      setScreenshot(file);
      setError("");
    }
  };

  const handleBankSelect = (bankName) => {
    setSelectedBank(bankName);
    const bank = bankDetails.find(b => b.bank_name === bankName);
    if (bank) {
      setSelectedBankDetails({
        accountName: bank.account_holder_name,
        accountNumber: bank.account_number,
        ifscCode: bank.ifsc_code,
        upiId: bank.upi_id
      });
    }
    setIsOpenNetwork(false);
  };

  const handleCopyAddress = (field, value) => {
    navigator.clipboard.writeText(value);
    setCopiedStates(prev => ({ ...prev, [field]: true }));
    setTimeout(() => setCopiedStates(prev => ({ ...prev, [field]: false })), 2000);
  };

  useEffect(() => {
    setError("");
    setShowTransactionForm(false);
    setTransactionId("");
    setScreenshot(null);
    setPhoneNumber(""); // Reset phone number
    setCCInitiate(false);
    setState(prev => ({ ...prev, amount: "" }));
  }, [selectedCoin]);

  const handleCopy = () => {
    navigator.clipboard.writeText(state.depositAddress);
    setState(prev => ({ ...prev, isCopied: true }));
    setTimeout(() => setState(prev => ({ ...prev, isCopied: false })), 2000);
    sendNotfication("Copied", "success", "bottom-left", true);
  };

  const loadCoinId = useCallback(async (coin) => {
    if (coin.toUpperCase() === 'INR' || isFiatCurrency(coin)) return;

    setState(prev => ({
      ...prev,
      currentCoin: coin,
      coinId: "",
      networks: [],
      selectedNetwork: "",
      isLoading: true,
      coinName: "",
      logoUrl: ""
    }));

    try {
      const token = storage.getKey('token');
      
      // Check if token exists before making the API call
      if (!token) {
        console.log('🔐 No token found, skipping coin details fetch');
        setState(prev => ({ ...prev, isLoading: false }));
        return;
      }

      console.log('🔐 Fetching coin details with token:', !!token);

      const response = await axios.post('https://api.boss707.com/getCoinDetails', 
        { symbol: coin.toUpperCase() },
        {
          headers: {
            'Content-Type': 'application/json',
            'Authorization': `Bearer ${token}`
          }
        }
      );
      const { coinId, networks, coinFullName, logoUrl } = response.data;
      const networksList = Object.keys(networks);
      setSelectedNetwork(networksList[0]);

      setState(prev => ({
        ...prev,
        coinId,
        networks: networksList,
        selectedNetwork: networksList[0],
        isLoading: false,
        coinName: coinFullName,
        logoUrl
      }));
    } catch (error) {
      console.error("Error fetching coin details:", error);
      setState(prev => ({ ...prev, isLoading: false }));
    }
  }, []);

  const initiateDeposit = async () => {
    if (!state.amount || state.amount <= 0) {
      sendNotfication("Please enter a correct amount.", "error", "top-right");
      return;
    }
    if (!selectedNetwork) {
      sendNotfication("Please Choose Network.", "error", "top-right");
      return;
    }
    setCCInitiate(true);
    setState(prev => ({ ...prev, isLoading: true, qrString: "" }));

    const payload = {
      coinId: state.coinId,
      price: state.amount,
      orderId: `order${Date.now()}`,
      chain: selectedNetwork,
      generateCheckoutURL: true,
      returnUrl: "https://api.boss707.com/",
      userid: getUID().toString()
    };

    try {
      const response = await axios.post('https://api.boss707.com/createDeposit', payload);
      const { data } = response.data;
      if (data.address) {
        setState(prev => ({
          ...prev,
          depositAddress: data.address,
          depositAmount: state.amount,
          showDepositDetails: true,
          isLoading: false,
          qrString: `${data.address}`
        }));
      } else {
        sendNotfication("Deposit initiation failed", "error", "top-right");
        setState(prev => ({ ...prev, isLoading: false }));
      }
    } catch (error) {
      console.error("Error initiating deposit:", error);
      sendNotfication("Error initiating deposit", "error", "top-right");
      setState(prev => ({ ...prev, isLoading: false }));
    }
  };

  const handleUPI1Payment = async () => {



    if (!state.amount || parseFloat(state.amount) < 300) {
      sendNotfication("Please enter an amount equal to or greater than ₹300.", "error", "top-right");
      return;
    }

    setState(prev => ({ ...prev, isLoading: true }));

    try {
      const response = await axios.post('https://api.boss707.com/remotes/create-deposit', {
        amount: state.amount,
        currency: 'INR',
        custom_user_id: getUID(),
        custom_transaction_id: `TX_${Date.now()}`,
        payment_system: "phonepe",
        webhook_id: 2472417,
        return_url: 'https://api.boss707.com',
      });

      const { success, url, order_id } = response.data;

      if (success) {
        setState(prev => ({
          ...prev,
          depositAmount: state.amount,
          orderId: order_id,
          isLoading: false,
          iframeUrl: url, // Store the URL for iframe
          showIframe: true // Show the iframe
        }));

      } else {
        sendNotfication("Payment initiation failed", "error", "top-right");
      }
    } catch (error) {
      console.error("Error initiating INR payment:", error);
      if (!state.amount || parseFloat(state.amount) < 300) {
        sendNotfication("Please enter amount equal to or greater than ₹300.", "error", "top-right");
      } else {
        sendNotfication(error.message, "error", "top-right");
      }
    } finally {
      setState(prev => ({ ...prev, isLoading: false }));
    }
  };

  const handleIMPSPayment = async () => {
    if (!state.amount || parseFloat(state.amount) < 300) {
      sendNotfication("Please enter an amount equal to or greater than ₹300.", "error", "top-right");
      return;
    }

    setState(prev => ({ ...prev, isLoading: true }));

    try {
      const response = await axios.post('https://api.boss707.com/remotes/create-deposit', {
        amount: state.amount,
        currency: 'INR',
        custom_user_id: getUID(),
        custom_transaction_id: `TX_${Date.now()}`,
        payment_system: "imps",
        webhook_id: 2472417,
        return_url: 'https://api.boss707.com',
      });

      const { success, url, order_id } = response.data;

      if (success) {

        setState(prev => ({
          ...prev,
          depositAmount: state.amount,
          orderId: order_id,
          isLoading: false,
          iframeUrl: url, // Store the URL for iframe
          showIframe: true // Show the iframe
        }));

      } else {
        sendNotfication("Payment initiation failed", "error", "top-right");
      }
    } catch (error) {
      console.error("Error initiating INR payment:", error);
      if (!state.amount || parseFloat(state.amount) < 300) {
        sendNotfication("Please enter amount equal to or greater than ₹300.", "error", "top-right");
      } else {
        sendNotfication(error.message, "error", "top-right");
      }
    } finally {
      setState(prev => ({ ...prev, isLoading: false }));
    }
  };
  
  const [upiMethod, setUpiMethod] = useState('upi1');
  const handleUPI2Payment = async () => {
    // if (state.currentCoin !== 'INR') return;
    if (!state.amount || parseFloat(state.amount) < 100) {
      sendNotfication("Please enter an amount ≥ ₹100.", "error", "top-right");
      return;
    }
    if (!phoneNumber || phoneNumber.length < 10) {
      sendNotfication("Please enter a valid phone number.", "error", "top-right");
      return;
    }

    setState(prev => ({ ...prev, isLoading: true }));
    try {
      const response = await axios.post('https://api.boss707.com/api/payments/payin/initiate', {
        userId: getUID(),
        currency: "INR",
        out_trade_no: `TX${Date.now()}`,
        pay_type: "UPI",
        money: state.amount,
        attach: "",
        notify_url: "https://api.boss707.com/api/payments/payin/callback",
        returnUrl: "https://api.boss707.com",
        phone: phoneNumber
      });

      const { code, data } = response.data;
      if (code === 0) {
        setState(prev => ({
          ...prev,
          depositAmount: state.amount,
          orderId: data.transaction_Id,
          isLoading: false,
          iframeUrl: data.url, // Store the URL for iframe
          showIframe: true // Show the iframe
        }));
      } else {
        sendNotfication("Payment initiation failed", "error", "top-right");
      }
    } catch (error) {
      console.error("Error initiating UPI payment:", error);
      sendNotfication("Error in payment", "error", "top-right");
    } finally {
      setState(prev => ({ ...prev, isLoading: false }));
    }
  };

  const handleSubmitTransactionFiat = async () => {
    if (!transactionId || !screenshot) {
      setError("Please fill all required fields");
      return;
    }
    if (!state.amount || parseFloat(state.amount) <= 0) {
      setError("Please enter a valid amount");
      return;
    }

    const formData = new FormData();
    formData.append('user_id', getUID());
    formData.append('amount', state.amount);
    formData.append('transaction_id', transactionId);
    formData.append('screenshot', screenshot);
    formData.append('currency', selectedCoin);
    if (selectedBank) {
      formData.append('bank_name', selectedBank);
      formData.append('account_number', selectedBankDetails.accountNumber);
      formData.append('ifsc_code', selectedBankDetails.ifscCode);
      formData.append('account_holder_name', selectedBankDetails.accountName);
    }

    try {
      const response = await axios.post('https://api.boss707.com/api/deposits/create', formData, {
        headers: { 'Content-Type': 'multipart/form-data' }
      });
      if (response.data.success) {
        sendNotfication("Deposit request created successfully", "success", "top-right");
        setShowTransactionForm(false);
        setTransactionId("");
        setScreenshot(null);
        setCCInitiate(false);
        setState(prev => ({ ...prev, amount: "" }));
      }
    } catch (error) {
      console.error("Error submitting transaction:", error);
      setError("Failed to submit deposit request");
    }
  };

  const handleCoinSelect = async (coinName) => {
    setSelectedNetwork('');
    setState(prev => ({
      ...prev,
      coinId: "",
      networks: [],
      selectedNetwork: "",
      coinName: "",
      logoUrl: ""
    }));
    setCCInitiate(false);
    if (storage.getKey('coin') === coinName) {
      setIsOpen(false);
      return;
    }

    storage.setKey('coin', coinName);
    setSelectedCoin(coinName);
    loadCoinId(coinName);
    setIsOpen(false);
  };
  // Archive Management
  const showArchive = useCallback(() => {
    if (state.slide) Event.emit("deposit_archive_back");

    setState(prev => ({ ...prev, slide: !prev.slide }));

    wait(300).then(() => {
      setState(prev => ({ ...prev, deposit: !prev.deposit }));
    });

    wait(300).then(() => {
      loadCoinId("BTC");
    });
  }, [state.slide]);

  const setUpCoins = useCallback(() => {
    if (isMobile()) {
      setState(prev => ({ ...prev, height: 685, margin: "mt-1" }));
    }

    const newList = coins.reduce((acc, item, i) => {
      if (item.preffix === "NC") return acc;

      const credit = forceSatoshiFormat(
        state.credits[__.lowerCase(item.preffix)]
      );

      const listItem = (
        <Dropdown.Item
          key={__.toString(i)}
          as="button"
          className="animated fadeIn"
          onClick={() => loadCoinId(item.preffix)}
        >
          <span>
            <img
              src={`/assets/images/${item.image}`}
              className="mr-1 img-fluid mini-coin"
              alt="Coin"
            />
            {item.preffix}
          </span>
        </Dropdown.Item>
      );

      return [...acc, listItem];
    }, []);

    setState(prev => ({ ...prev, list: newList }));
  }, [state.credits, isMobile]);

  // Socket and Credit Management
  const getCreditAndCoins = useCallback((data) => {
    const { credit } = data;
    setState(prev => ({ ...prev, credits: credit }));
    setUpCoins();
  }, []);
  // Initial Setup Effect
  useEffect(() => {
    socket.emit(C.CREDIT, encode({ coin: storage.getKey('coin') || "BTC" }));
    socket.on(C.CREDIT, (data) => getCreditAndCoins(decode(data)));

    socket.on("deposit_archive", showArchive);

    wait(400).then(() => {
      loadCoinId(storage.getKey('coin') || "BTC");
    });

    return () => {
      socket.off(C.CREDIT);
    };
  }, [storage.getKey('coin') || "BTC", getCreditAndCoins, loadCoinId, showArchive]);

  // Sync selectedCoin with SkylaCredit via coin_changed event
  useEffect(() => {
    const handleCoinChanged = (coinName) => {
      setSelectedCoin(coinName); // Update selectedCoin when coin changes in SkylaCredit
      setState(prev => ({ ...prev, currentCoin: coinName })); // Update state.currentCoin
      loadCoinId(coinName); // Load coin details for the new coin
    };

    socket.on('coin_changed', handleCoinChanged);

    // Initialize with stored coin
    const storedCoin = storage.getKey('coin') || "BTC";
    setSelectedCoin(storedCoin);
    setState(prev => ({ ...prev, currentCoin: storedCoin }));
    loadCoinId(storedCoin);


  }, [loadCoinId]);

   const handleUPI3Payment = async () => {
    if (!state.amount || parseFloat(state.amount) < 1000) {
      sendNotfication("Please enter an amount equal to or greater than ₹1000.", "error", "top-right");
      return;
    }

    setState(prev => ({ ...prev, isLoading: true }));

    try {
      const name = localStorage.getItem("name");
      const response = await axios.post('https://api.boss707.com/cricpay/payment-request', {
        amount: state.amount,
        userId: getUID(),
       paymentMethod: "PS002",
       userName:"jhonSon"
      });

   
      const { status, accessURL, transactionCode } = response.data;

      if (status==="Success") {

        setState(prev => ({
          ...prev,
          depositAmount: state.amount,
          orderId: transactionCode,
          isLoading: false,
          iframeUrl: accessURL, // Store the URL for iframe
          showIframe: true // Show the iframe
        }));

      } else {
        sendNotfication("Payment initiation failed", "error", "top-right");
      }
    } catch (error) {
      console.error("Error initiating INR payment:", error);
      if (!state.amount || parseFloat(state.amount) < 1000) {
        sendNotfication("Please enter amount equal to or greater than ₹1000.", "error", "top-right");
      } else {
        sendNotfication(error.message, "error", "top-right");
      }
    } finally {
      setState(prev => ({ ...prev, isLoading: false }));
    }
  };

   const handleNetBankingPayment = async () => {
    if (!state.amount || parseFloat(state.amount) < 1000) {
      sendNotfication("Please enter an amount equal to or greater than ₹1000.", "error", "top-right");
      return;
    }

    setState(prev => ({ ...prev, isLoading: true }));

    try {
      const name = localStorage.getItem("name");
      const response = await axios.post('https://api.boss707.com/cricpay/payment-request', {
          amount: state.amount,
        userId: getUID(),
       paymentMethod: "PS003",
        userName: "jhonSon"
      });

      const { status, accessURL, transactionCode } = response.data;

      if (status==="Success") {

        setState(prev => ({
          ...prev,
          depositAmount: state.amount,
          orderId: transactionCode,
          isLoading: false,
          iframeUrl: accessURL, // Store the URL for iframe
          showIframe: true // Show the iframe
        }));

      } else {
        sendNotfication("Payment initiation failed", "error", "top-right");
      }
    } catch (error) {
      console.error("Error initiating INR payment:", error);
      if (!state.amount || parseFloat(state.amount) < 1000) {
        sendNotfication("Please enter amount equal to or greater than ₹1000.", "error", "top-right");
      } else {
        sendNotfication(error.message, "error", "top-right");
      }
    } finally {
      setState(prev => ({ ...prev, isLoading: false }));
    }
  };

  const filteredCoins = coins
    .filter(coin => {
      if (!activeCoins.includes(coin.id.toString())) return false;
      if (coin.preffix === 'BJT' || coin.preffix === 'SC') return false;
      return coin.preffix.toLowerCase().includes(searchTerm.toLowerCase());
    })
    .sort((a, b) => (a.preffix === 'BTC' ? -1 : b.preffix === 'BTC' ? 1 : 0));

  const selectedCoinData = coins.find(coin => coin.preffix === selectedCoin);

  const renderPaymentOptions = () => {
    if (selectedCoin === 'INR') {
      return (
        <PaymentMethodContainer>
          <MethodSelector>
            <MethodButton
              selected={paymentMode === 'automatic'}
              onClick={() => setPaymentMode('automatic')}
            >
              Automatic Payment
            </MethodButton>
            <MethodButton
              selected={paymentMode === 'manual'}
              onClick={() => setPaymentMode('manual')}
            >
              Manual Payment
            </MethodButton>
          </MethodSelector>

          {paymentMode === 'automatic' && (
            <>
              {/* UPI Method Selector */}
              <MethodSelector style={{ marginTop: '10px' }}>
                <MethodButton
                  selected={upiMethod === 'upi1'}
                  onClick={() => setUpiMethod('upi1')}
                >
                  UPI 1
                </MethodButton>
                <MethodButton
                  selected={upiMethod === 'upi2'}
                  onClick={() => setUpiMethod('upi2')}
                >
                  UPI 2
                </MethodButton>
                 <MethodButton
                  selected={upiMethod === 'upi3'}
                  onClick={() => setUpiMethod('upi3')}
                >
                 UPI 3
                </MethodButton>
                <MethodButton
                  selected={upiMethod === 'imps'}
                  onClick={() => setUpiMethod('imps')}
                >
                  IMPS
                </MethodButton>
                 <MethodButton
                  selected={upiMethod === 'netbanking'}
                  onClick={() => setUpiMethod('netbanking')}
                >
                 Netbanking
                </MethodButton>
              </MethodSelector>

              {/* Show phone number input only for UPI 2 */}
              {upiMethod === 'upi2' && (
                <Input
                  value={phoneNumber}
                  onChange={(e) => setPhoneNumber(e.target.value)}
                  placeholder="Enter Phone Number"
                  style={{ marginTop: '10px' }}
                />
              )}
            </>
          )}

          {paymentMode === 'automatic' ? (
            <DepositButton
              onClick={upiMethod === 'upi1' ? handleUPI1Payment :
                upiMethod === 'imps' ? handleIMPSPayment :
                upiMethod === 'upi3' ? handleUPI3Payment :
                upiMethod === 'netbanking' ? handleNetBankingPayment :
                  handleUPI2Payment}
            >
              Proceed to Automatic Payment
            </DepositButton>
          ) : (
            <DepositButton
              onClick={() => {
                setShowTransactionForm(true);
                setCCInitiate(true); // Fixed typo in setCCInitiate
              }}
            >
              Proceed to Payment
            </DepositButton>
          )}
        </PaymentMethodContainer>
      );
    } else if (isFiatCurrency(selectedCoin)) {
      return (
        <DepositButton
          onClick={() => {
            setShowTransactionForm(true);
            setCCInitiate(true);
          }}
        >
          Proceed to Payment
        </DepositButton>
      );
    } else {
      // For crypto currencies
      return (
        !ccInitiate ?
          <DepositButton onClick={initiateDeposit}>
            Proceed with Deposit
          </DepositButton> : <></>
      );
    }
  };

  useEffect(() => {
    socket.emit(C.CREDIT, encode({ coin: selectedCoin }));
    socket.on(C.CREDIT, (data) => {
      const decodedData = decode(data);
      setState(prev => ({ ...prev, credits: decodedData.credit }));
    });
    wait(400).then(() => loadCoinId(selectedCoin));
    return () => socket.off(C.CREDIT);
  }, [selectedCoin]);

 const getAuthToken = () => {
  return localStorage.getItem('token');
};
useEffect(() => {
  fetchBonusConfig();
}, []);

const fetchBonusConfig = async () => {
  try {
   const token = localStorage.getItem('token');

    const response = await fetch('https://api.boss707.com/api/bonus/config-public', {
      method: 'GET',
      headers: {
        Authorization: `Bearer ${token}`,
        'Content-Type': 'application/json',
      },
    });

    const data = await response.json();
    setBonusConfig(data);
  } catch (error) {
    setBonusConfig([]);
  }
};


  const [showCasinoInfo, setShowCasinoInfo] = useState(false);
  const [showSportsInfo, setShowSportsInfo] = useState(false);

  const casinoBonus = bonusConfig.find(b => b.bonus_type === 'casino') || {};
  const sportsBonus = bonusConfig.find(b => b.bonus_type === 'sports') || {};

  const CasinoCard = (
    <BonusCardContainer
      className={selectedBonusCard === 'casino' ? 'selected' : ''}
      onClick={() => setSelectedBonusCard('casino')}
    >
    
      <BonusSVGWrapper>
        <SmallImagePlaceholder src="/assets/images/bonus/wheel.webp" alt="Casino Bonus" />
      </BonusSVGWrapper>
      <BonusCardSectionTitle>Casino</BonusCardSectionTitle>
        <InfoIconButton onClick={e => { e.stopPropagation(); setShowCasinoInfo(true); }}>
        <InfoIcon size={16} />
      </InfoIconButton>
      
      {showCasinoInfo && (
        <BonusInfoModalOverlay onClick={() => setShowCasinoInfo(false)}>
          <BonusInfoModalContent onClick={e => e.stopPropagation()}>
            <BonusInfoModalClose onClick={() => setShowCasinoInfo(false)}>&times;</BonusInfoModalClose>
          
            <div style={{ fontWeight: 700, fontSize: 22, marginBottom: 18 }}>Casino Bonus</div>
            <ul style={{ paddingLeft: 0, listStyle: 'none', color: '#fff', fontSize: 16 }}>
              <li style={{ marginBottom: 12, display: 'flex', alignItems: 'center' }}>
                <span style={{ color: '#F79C03', marginRight: 10, fontSize: 18 }}>•</span>
                Enjoy a 300% bonus on your casino deposit
              </li>
              <li style={{ marginBottom: 12, display: 'flex', alignItems: 'center' }}>
                <span style={{ color: '#F79C03', marginRight: 10, fontSize: 18 }}>•</span>
               Minimum deposit required: $10
              </li>
              <li style={{ marginBottom: 12, display: 'flex', alignItems: 'center' }}>
                <span style={{ color: '#F79C03', marginRight: 10, fontSize: 18 }}>•</span>
                Wagering requirement: 30x the bonus amount
              </li>
              <li style={{ marginBottom: 12, display: 'flex', alignItems: 'center' }}>
                <span style={{ color: '#F79C03', marginRight: 10, fontSize: 18 }}>•</span>
                Available on selected slot and table games
              </li>
              <li style={{ marginBottom: 0, display: 'flex', alignItems: 'center' }}>
                <span style={{ color: '#F79C03', marginRight: 10, fontSize: 18 }}>•</span>
                Claimable once per user per promotion
              </li>
            </ul>
          </BonusInfoModalContent>
        </BonusInfoModalOverlay>
      )}
    </BonusCardContainer>
  );

  const SportsCard = (
    <BonusCardContainer
      className={selectedBonusCard === 'sports' ? 'selected' : ''}
      onClick={() => setSelectedBonusCard('sports')}
    >
  
       <BonusSVGWrapper>
        <SmallImagePlaceholder src="/assets/images/bonus/cup-1.webp" alt="Sports Bonus" />
      </BonusSVGWrapper>
      <BonusCardSectionTitle>Sports</BonusCardSectionTitle>
          <InfoIconButton onClick={e => { e.stopPropagation(); setShowSportsInfo(true); }}>
        <InfoIcon size={16} />
      </InfoIconButton>
     
      {showSportsInfo && (
        <BonusInfoModalOverlay onClick={() => setShowSportsInfo(false)}>
          <BonusInfoModalContent onClick={e => e.stopPropagation()}>
            <BonusInfoModalClose onClick={() => setShowSportsInfo(false)}>&times;</BonusInfoModalClose>
            <div style={{ fontWeight: 700, fontSize: 22, marginBottom: 18 }}>Sports Bonus</div>
         
            <ul style={{ paddingLeft: 0, listStyle: 'none', color: '#fff', fontSize: 16 }}>
              <li style={{ marginBottom: 12, display: 'flex', alignItems: 'center' }}>
                <span style={{ color: '#F79C03', marginRight: 10, fontSize: 18 }}>•</span>
                Get a 300% bonus on your sports deposit
              </li>
              <li style={{ marginBottom: 12, display: 'flex', alignItems: 'center' }}>
                <span style={{ color: '#F79C03', marginRight: 10, fontSize: 18 }}>•</span>
                Minimum deposit required: $10
              </li>
              <li style={{ marginBottom: 12, display: 'flex', alignItems: 'center' }}>
                <span style={{ color: '#F79C03', marginRight: 10, fontSize: 18 }}>•</span>
                Wagering requirement: 30x the bonus amount
              </li>
              <li style={{ marginBottom: 12, display: 'flex', alignItems: 'center' }}>
                <span style={{ color: '#F79C03', marginRight: 10, fontSize: 18 }}>•</span>
                Available for all registered users
              </li>
              <li style={{ marginBottom: 0, display: 'flex', alignItems: 'center' }}>
                <span style={{ color: '#F79C03', marginRight: 10, fontSize: 18 }}>•</span>
                Applies to selected sports betting markets
              </li>
            </ul>
          </BonusInfoModalContent>
        </BonusInfoModalOverlay>
      )}
    </BonusCardContainer>
  );

  const BonusCards = (
    <BonusCardsWrapper>
      {CasinoCard}
      {SportsCard}
    </BonusCardsWrapper>
  );

  // Add a Get Bonus button before rendering BonusCards
  const handleGetBonusClick = () => setShowBonusCards((prev) => !prev);

  return (
    <Container>
      <TokenListContainer>
        {tokens.map(token => (
          <TokenButton key={token.id}>
            <TokenIcon src={token.icon} alt={`${token.name} icon`} />
            <TokenName>{token.name}</TokenName>
          </TokenButton>
        ))}
      </TokenListContainer>

      {
        paymentMode === "automatic" && selectedCoin === "INR" ? (
          <div style={{ display: "flex", gap: "5px", width: "100%" }}>
            <div style={{ flex: '1' }}>
              <div style={{ color: "#B3BEC1", margin: "4px 0" }}>Deposit Currency</div>
              <DropdownContainer>
                <DropdownButton onClick={() => setIsOpen(!isOpen)}>
                  <div style={{ display: 'flex', alignItems: 'center', gap: '12px' }}>
                    <CoinImage src={`/assets/images/${selectedCoinData?.image}`} alt={selectedCoin} />
                    <CoinName>{selectedCoin}</CoinName>
                  </div>
                  <ChevronDown size={20} />
                </DropdownButton>
                {isOpen && (
                  <DropdownPanel onClick={e => e.stopPropagation()}>
                    <SearchContainer>
                      <SearchIcon><Search size={20} /></SearchIcon>
                      <SearchInput
                        placeholder="Search coins..."
                        value={searchTerm}
                        onChange={(e) => setSearchTerm(e.target.value)}
                        autoFocus
                      />
                    </SearchContainer>
                    {filteredCoins.map((coin) => (
                      <CoinOption
                        key={coin.id}
                        onClick={() => handleCoinSelect(coin.preffix)}
                        isSelected={selectedCoin === coin.preffix}
                      >
                        <CoinImage src={`/assets/images/${coin.image}`} alt={coin.preffix} />
                        <CoinName>{coin.preffix}</CoinName>
                      </CoinOption>
                    ))}
                  </DropdownPanel>
                )}
              </DropdownContainer>
            </div>
     

            {isFiatCurrency(selectedCoin) ? (
              <div style={{ flex: '1' }}>
                <div style={{ color: "#B3BEC1", margin: "4px 0" }}>Choose Bank</div>
                <DropdownContainer>
                  <DropdownButton onClick={() => setIsOpenNetwork(!isOpenNetwork)}>
                    <div style={{ display: 'flex', alignItems: 'center', gap: '12px' }}>
                      <CoinName>{selectedBank || "Select Bank"}</CoinName>
                    </div>
                    <ChevronDown size={20} />
                  </DropdownButton>
                  {isOpenNetwork && (
                    <DropdownPanel onClick={e => e.stopPropagation()}>
                      {bankDetails.map((bank) => (
                        <CoinOption
                          key={bank.id}
                          onClick={() => handleBankSelect(bank.bank_name)}
                          isSelected={selectedBank === bank.bank_name}
                        >
                          <CoinName>{bank.bank_name}</CoinName>
                        </CoinOption>
                      ))}
                    </DropdownPanel>
                  )}
                </DropdownContainer>
              </div>
            ) : (
              state.networks?.length > 0 && (
                <div style={{ flex: '1' }}>
                  <div style={{ color: "#B3BEC1", margin: "4px 0" }}>Choose Network</div>
                  <DropdownContainer>
                    <DropdownButton onClick={() => setIsOpenNetwork(!isOpenNetwork)}>
                      <CoinName>{selectedNetwork}</CoinName>
                      <ChevronDown size={20} />
                    </DropdownButton>
                    {isOpenNetwork && (
                      <DropdownPanel onClick={e => e.stopPropagation()}>
                        {state.networks.map((network) => (
                          <CoinOption
                            key={network}
                            onClick={() => {
                              setSelectedNetwork(network);
                              setIsOpenNetwork(false);
                            }}
                            isSelected={selectedNetwork === network}
                          >
                            <CoinName>{network}</CoinName>
                          </CoinOption>
                        ))}
                      </DropdownPanel>
                    )}
                  </DropdownContainer>
                </div>
              )
            )}
          </div>
        ) : (
          <div style={{ display: "flex", gap: "5px", width: "100%" }}>
            <div style={{ flex: '1' }}>
              <div style={{ color: "#B3BEC1", margin: "4px 0" }}>Deposit Currency</div>
              <DropdownContainer>
                <DropdownButton onClick={() => setIsOpen(!isOpen)}>
                  <div style={{ display: 'flex', alignItems: 'center', gap: '12px' }}>
                    <CoinImage src={`/assets/images/${selectedCoinData?.image}`} alt={selectedCoin} />
                    <CoinName>{selectedCoin}</CoinName>
                  </div>
                  <ChevronDown size={20} />
                </DropdownButton>
                {isOpen && (
                  <DropdownPanel onClick={e => e.stopPropagation()}>
                    <SearchContainer>
                      <SearchIcon><Search size={20} /></SearchIcon>
                      <SearchInput
                        placeholder="Search coins..."
                        value={searchTerm}
                        onChange={(e) => setSearchTerm(e.target.value)}
                        autoFocus
                      />
                    </SearchContainer>
                    {filteredCoins.map((coin) => (
                      <CoinOption
                        key={coin.id}
                        onClick={() => handleCoinSelect(coin.preffix)}
                        isSelected={selectedCoin === coin.preffix}
                      >
                        <CoinImage src={`/assets/images/${coin.image}`} alt={coin.preffix} />
                        <CoinName>{coin.preffix}</CoinName>
                      </CoinOption>
                    ))}
                  </DropdownPanel>
                )}
              </DropdownContainer>
            </div>

            {isFiatCurrency(selectedCoin) ? (
              <div style={{ flex: '1' }}>
                <div style={{ color: "#B3BEC1", margin: "4px 0" }}>Choose Bank</div>
                <DropdownContainer>
                  <DropdownButton onClick={() => setIsOpenNetwork(!isOpenNetwork)}>
                    <div style={{ display: 'flex', alignItems: 'center', gap: '12px' }}>
                      <CoinName>{selectedBank || "Select Bank"}</CoinName>
                    </div>
                    <ChevronDown size={20} />
                  </DropdownButton>
                  {isOpenNetwork && (
                    <DropdownPanel onClick={e => e.stopPropagation()}>
                      {bankDetails.map((bank) => (
                        <CoinOption
                          key={bank.id}
                          onClick={() => handleBankSelect(bank.bank_name)}
                          isSelected={selectedBank === bank.bank_name}
                        >
                          <CoinName>{bank.bank_name}</CoinName>
                        </CoinOption>
                      ))}
                    </DropdownPanel>
                  )}
                </DropdownContainer>
              </div>
            ) : (
              state.networks?.length > 0 && (
                <div style={{ flex: '1' }}>
                  <div style={{ color: "#B3BEC1", margin: "4px 0" }}>Choose Network</div>
                  <DropdownContainer>
                    <DropdownButton onClick={() => setIsOpenNetwork(!isOpenNetwork)}>
                      <CoinName>{selectedNetwork}</CoinName>
                      <ChevronDown size={20} />
                    </DropdownButton>
                    {isOpenNetwork && (
                      <DropdownPanel onClick={e => e.stopPropagation()}>
                        {state.networks.map((network) => (
                          <CoinOption
                            key={network}
                            onClick={() => {
                              setSelectedNetwork(network);
                              setIsOpenNetwork(false);
                            }}
                            isSelected={selectedNetwork === network}
                          >
                            <CoinName>{network}</CoinName>
                          </CoinOption>
                        ))}
                      </DropdownPanel>
                    )}
                  </DropdownContainer>
                </div>
              )
            )}
          </div>
        )
      }
              {/* Add a Get Bonus button before rendering BonusCards */}
              <div style={{ width: '100%', display: 'flex', justifyContent: 'left', margin: '16px 0' }}>
        <DepositButton style={{ maxWidth: 150 }} onClick={handleGetBonusClick}>
          {showBonusCards ? 'Get Bonus' : 'Get Bonus'}
        </DepositButton>
      </div>
      {showBonusCards && BonusCards}

      {/* Add the new note container right here */}
      <div style={{ margin: "12px 0" }}>
        <WarningContainer>
          <Info color="#ab1ac6" />
          <WarningText>Do not deposit any currency other than {selectedCoin}</WarningText>
        </WarningContainer>

        <NoteContainer>
          <NoteIcon size={20} />
          <NoteText>
            Usually deposits are instant but in some cases {selectedCoin === "INR" ? "INR" : selectedCoin} deposits can take up to 2hrs.
            If you haven't received your deposit amount, please contact customer support.
          </NoteText>
        </NoteContainer>
      </div>

      {state.isLoading ? (
        <SpinnerWrapper className="p-4">
          <Spinner />
          <LoadingText>Loading</LoadingText>
        </SpinnerWrapper>
      ) : (
        ccInitiate && (
          <div style={{ margin: "12px 0" }}>
            {isFiatCurrency(selectedCoin) ? (
              <div style={{ display: 'flex', alignItems: 'center', gap: '10px' }}>
                <QRCodeContainer>
                  {selectedCoin === "INR" && selectedBankDetails.qr_image ? (
                    <img src={selectedBankDetails.qr_image} alt="QR Code" />
                  ) : selectedCoin === "INR" ? (
                    <img
                      src="data:image/png;base64,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"
                      alt="QR Code"
                    />
                  ) : null}
                </QRCodeContainer>

                <ContentContainer>

                  {/* <Label>{selectedBank ? `${selectedBank} Bank Details` : "Bank Details"}</Label> */}
                  {selectedBankDetails?.accountName && (
                    <AddressContainer>
                      <span>Account Holder Name: {selectedBankDetails.accountName}</span>
                      <CopyButton onClick={() => handleCopyAddress('accountName', selectedBankDetails.accountName)}>
                        {copiedStates.accountName ? <CheckCircle size={18} /> : <Copy size={18} />}
                      </CopyButton>
                    </AddressContainer>
                  )}
                  {selectedBankDetails?.accountNumber && (
                    <AddressContainer>
                      <span>Account Number: {selectedBankDetails.accountNumber}</span>
                      <CopyButton onClick={() => handleCopyAddress('accountNumber', selectedBankDetails.accountNumber)}>
                        {copiedStates.accountNumber ? <CheckCircle size={18} /> : <Copy size={18} />}
                      </CopyButton>
                    </AddressContainer>
                  )}
                  {selectedBankDetails?.ifscCode && (
                    <AddressContainer>
                      <span>IFSC Code: {selectedBankDetails.ifscCode}</span>
                      <CopyButton onClick={() => handleCopyAddress('ifscCode', selectedBankDetails.ifscCode)}>
                        {copiedStates.ifscCode ? <CheckCircle size={18} /> : <Copy size={18} />}
                      </CopyButton>
                    </AddressContainer>
                  )}
                  {selectedBankDetails?.upiId && (
                    <AddressContainer>
                      <span>UPI ID: {selectedBankDetails.upiId}</span>
                      <CopyButton onClick={() => handleCopyAddress('upiId', selectedBankDetails.upiId)}>
                        {copiedStates.upiId ? <CheckCircle size={18} /> : <Copy size={18} />}
                      </CopyButton>
                    </AddressContainer>
                  )}
                </ContentContainer>
              </div>
            ) : (
              <div style={{ display: "flex", alignItems: '', gap: '10px' }}>
                <QRCode size={150} logoWidth={150} value={state.qrString} />
                <ContentContainer>
                  <Label>Deposit address</Label>
                  <AddressContainer>
                    <span>{state.depositAddress}</span>
                    <CopyButton onClick={handleCopy}>
                      {state.isCopied ? <CheckCircle size={18} /> : <Copy size={18} />}
                    </CopyButton>
                  </AddressContainer>
                </ContentContainer>
              </div>
            )}
          </div>
        )
      )}

      <div style={{ display: "flex", gap: "10px", width: "100%", flexDirection: 'column', marginTop: "10px" }}>
        <Input
          value={state.amount}
          onChange={(e) => {
            const newAmount = e.target.value;
            setState(prev => ({ ...prev, amount: newAmount }));
            setError("");
            if (state.currentCoin === "MVR" && newAmount && Number(newAmount) < 100) {
              setError("Minimum deposit for MVR is 100");
            }
          }}
          placeholder={state.currentCoin === "MVR" ? "Enter Deposit Amount (Minimum 100MVR)" : "Enter Deposit Amount"}
        />
        {error && <ErrorText>{error}</ErrorText>}
        {renderPaymentOptions()}
        {showTransactionForm && (
          <TransactionForm>
            <Input
              value={transactionId}
              onChange={(e) => setTransactionId(e.target.value)}
              placeholder="Enter Transaction/UTR Number"
            />
            <FileUploadContainer onClick={() => document.getElementById('fileInput').click()}>
              <Upload size={24} color="#41ED84" />
              <UploadText>{screenshot ? screenshot.name : 'Upload Payment Screenshot'}</UploadText>
              <input
                id="fileInput"
                type="file"
                accept="image/*"
                onChange={handleFileUpload}
                style={{ display: 'none' }}
              />
            </FileUploadContainer>
            <DepositButton onClick={handleSubmitTransactionFiat}>Submit</DepositButton>
          </TransactionForm>
        )}
      </div>

      {/* <div style={{ margin: "12px 0" }}>
        <WarningContainer>
          <Info color="#ab1ac6" />
          <WarningText>Do not deposit any currency other than {selectedCoin}</WarningText>
        </WarningContainer>
      </div> */}

      <div style={{ margin: "12px 0" }}>
        <WarningContainer>
          <Info color="#ab1ac6" />
          <WarningText>Do not deposit any currency other than {selectedCoin}</WarningText>
        </WarningContainer>

        {/* Add the new note container right here */}
        <NoteContainer>
          <NoteIcon size={20} />
          <NoteText>
            Usually deposits are instant but in some cases {selectedCoin === "INR" ? "INR" : selectedCoin} deposits can take up to 2hrs.
            If you haven't received your deposit amount, please contact customer support.
          </NoteText>
        </NoteContainer>
      </div>
      {state.showIframe && (
        <div
          style={{
            position: 'fixed',
            top: 0,
            left: 0,
            width: '100vw',
            height: '100vh',
            backgroundColor: 'rgba(0, 0, 0, 0.8)',
            zIndex: 1000,
            display: 'flex',
            flexDirection: 'column',
            alignItems: 'center',
            justifyContent: 'center',
          }}
        >
          <div
            style={{
              width: '100%',
              maxWidth: '600px',
              height: '600px',
              backgroundColor: '#fff',
              borderRadius: '8px',
              overflow: 'hidden',
              position: 'relative',
            }}
          >
            <button
              onClick={() => {
                setState((prev) => ({ ...prev, showIframe: false, iframeUrl: '' }));
                window.location.reload();
              }}
              style={{
                position: 'absolute',
                top: '12px',
                left: '12px',
                backgroundColor: 'rgba(255, 255, 255, 0.9)',
                border: 'none',
                borderRadius: '50%',
                padding: '10px',
                cursor: 'pointer',
                display: 'flex',
                alignItems: 'center',
                justifyContent: 'center',
                boxShadow: '0 2px 4px rgba(0, 0, 0, 0.1)',
                transition: 'background-color 0.2s, transform  reman 0.2s',
              }}
              onMouseEnter={(e) => (e.currentTarget.style.backgroundColor = 'rgba(255, 255, 255, 1)')}
              onMouseLeave={(e) => (e.currentTarget.style.backgroundColor = 'rgba(255, 255, 255, 0.9)')}
              onMouseDown={(e) => (e.currentTarget.style.transform = 'scale(0.95)')}
              onMouseUp={(e) => (e.currentTarget.style.transform = 'scale(1)')}
              aria-label="Back"
            >
              <svg
                width="20"
                height="20"
                viewBox="0 0 24 24"
                fill="none"
                stroke="#333"
                strokeWidth="2.5"
                strokeLinecap="round"
                strokeLinejoin="round"
              >
                <path d="M15 18l-6-6 6-6" />
              </svg>
            </button>
            <iframe
              src={state.iframeUrl}
              style={{ width: '100%', height: '100%', border: 'none' }}
              title="UPI Payment"
              sandbox="allow-same-origin allow-scripts allow-popups allow-forms"
            />
          </div>
        </div>
      )}

     
    </Container>
  );
};

export default SkylaDeposit;