// src/contexts/ExposureContext.jsx
import React, {
  createContext,
  useContext,
  useState,
  useCallback,
  useEffect
} from 'react';

const ExposureContext = createContext();
const API_ROOT = 'https://api.boss707.com/sportsbetting';

// helper: pull uid from cookies
const readUid = () => {
  const m = document.cookie.match(/(?:^|; )uid=([^;]+)/);
  return m ? decodeURIComponent(m[1]) : null;
};

const getAuthToken = () => {
  return localStorage.getItem('token');
};

// Check if user is authenticated
const isAuthenticated = () => {
  const token = getAuthToken();
  const logged = localStorage.getItem('logged');
  return token && logged === 'true';
};

export const ExposureProvider = ({ children }) => {
  const [exposures, setExposures] = useState({});
  const [balance, setBalance] = useState(0);
  const [isLoading, setIsLoading] = useState(false);
  const [error, setError] = useState(null);

  // fetch exposures + wallet from server
  const refreshAll = useCallback(async uuid => {
    if (!uuid) {
      console.log('🔐 ExposureProvider: No UUID provided, skipping API calls');
      return;
    }
    
    // Check authentication before making API calls
    if (!isAuthenticated()) {
      console.log('🔐 ExposureProvider: User not authenticated, skipping API calls');
      setError('Authentication required');
      return;
    }

    setIsLoading(true);
    setError(null);
    
    try {
      const token = getAuthToken();
      console.log('🔐 ExposureProvider: Making API calls with token:', !!token);
      
      // 1) exposures
      const eRes = await fetch(`${API_ROOT}/exposures/${uuid}`, {
        method: 'GET',
        headers: {
          Authorization: `Bearer ${token}`,
          'Content-Type': 'application/json',
        },
      });

      if (eRes.status === 401) {
        console.log('🔐 ExposureProvider: 401 Unauthorized for exposures');
        setError('Authentication required');
        return;
      }

      const eJ = await eRes.json();
      if (eJ.success) {
        setExposures(eJ.exposures);
        console.log('✅ ExposureProvider: Exposures loaded successfully');
      }

      // 2) wallet balance
      const wRes = await fetch(`${API_ROOT}/wallet/${uuid}`, {
        method: 'GET',
        headers: {
          Authorization: `Bearer ${token}`,
          'Content-Type': 'application/json',
        },
      });

      if (wRes.status === 401) {
        console.log('🔐 ExposureProvider: 401 Unauthorized for wallet');
        setError('Authentication required');
        return;
      }

      const wJ = await wRes.json();
      if (wJ.success) {
        const newBal = +wJ.balance;
        setBalance(newBal);
        // persist & notify header
        localStorage.setItem('credit', newBal.toString());
        window.dispatchEvent(
          new CustomEvent('credit_update', { detail: newBal })
        );
        console.log('✅ ExposureProvider: Wallet balance updated:', newBal);
      }
    } catch (err) {
      console.error('❌ ExposureProvider.refreshAll error:', err);
      setError(err.message || 'Failed to fetch data');
    } finally {
      setIsLoading(false);
    }
  }, []);

  // on mount: seed from storage, then refresh from API if authenticated
useEffect(() => {
  const timer = setTimeout(() => {
    const uid = readUid();
    console.log('🔐 ExposureProvider: Initializing with UID:', uid);
    
    if (uid && isAuthenticated()) {
      const stored = Number(localStorage.getItem('credit') || 0);
      setBalance(stored);
      console.log('🔐 ExposureProvider: User authenticated, refreshing data');
      refreshAll(uid);
    } else {
      console.log('🔐 ExposureProvider: User not authenticated, using default state');
      // Set default balance from storage even if not authenticated
      const stored = Number(localStorage.getItem('credit') || 0);
      setBalance(stored);
    }
  }, 2000); // 2 seconds delay

  return () => clearTimeout(timer);
}, [refreshAll]);

  // patch a single match's exposures
  const patchMatchExposure = (matchId, newTeams) => {
    setExposures(prev => ({
      ...prev,
      [matchId]: {
        match_title: prev[matchId]?.match_title || '',
        teams: {
          ...(prev[matchId]?.teams || {}),
          ...newTeams
        }
      }
    }));
  };

  // adjust balance by delta and persist/broadcast
  const patchBalance = delta => {
    setBalance(prev => {
      const updated = prev + delta;
      localStorage.setItem('credit', updated.toString());
      window.dispatchEvent(
        new CustomEvent('credit_update', { detail: updated })
      );
      return updated;
    });
  };

  // compute total exposure (sum of each match's max liability)
  const totalExposure = Object.values(exposures)
    .reduce((sum, m) => {
      const losses = Object.values(m.teams || {})
        .filter(v => v < 0)
        .map(v => Math.abs(v));
      const maxLoss = losses.length ? Math.max(...losses) : 0;
      return sum + maxLoss;
    }, 0)
    .toFixed(2);

  return (
    <ExposureContext.Provider
      value={{
        exposures,
        totalExposure,
        balance,
        patchMatchExposure,
        patchBalance,
        refreshAll,
        isLoading,
        error,
        isAuthenticated: isAuthenticated()
      }}
    >
      {children}
    </ExposureContext.Provider>
  );
};

export const useExposure = () => useContext(ExposureContext);
